/**************************************************
opengate Mac addr auth program

 module to control management database
 list of mac addresses
 list of usage log
 implemented with MySql 

Copyright (C) 2011 Opengate Project Team
Written by Yoshiaki Watanabe

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Email: watanaby@is.saga-u.ac.jp
**************************************************/
#include "opengatemmng.h"
#include <mysql.h>


MYSQL mysql;

/******************************************
initialize management db
******************************************/
int initMngDb(void){

  /* set parameters */
  char *server = GetConfValue("MySqlDb/Server");
  char *user =  GetConfValue("MySqlDb/User");
  char *password = GetConfValue("MySqlDb/Password");
  char *database = GetConfValue("MySqlDb/Database");
  my_bool reconnect;

/* initialize mysql */
  mysql_library_init(-1,NULL,NULL);
  if(mysql_init(&mysql)==NULL){
     err_msg("ERR at %s#%d: mysql init: %s",__FILE__,__LINE__,
	     mysql_error(&mysql));
     terminateProg(0);
  }

  /* Connect to database */
  if (!mysql_real_connect(&mysql, server,
			   user, password, database, 0, NULL, 0)) {
    err_msg("ERR at %s#%d: mysql connect: %s",__FILE__,__LINE__,
	     mysql_error(&mysql));
    terminateProg(0);
  }

  /* set auto-reconnect true */
  reconnect = TRUE;
  mysql_options(&mysql, MYSQL_OPT_RECONNECT, &reconnect);  

  return TRUE;
}

/******************************************
close management db
******************************************/
void closeMngDb(void){
  mysql_close(&mysql);
  mysql_library_end();
}


/******************************************
 get nic vendor from management db
******************************************/
int getNicVendorFromMngDb(char* macAddress, char* vendor, int bufferLength){

  MYSQL_RES *res;
  MYSQL_ROW row;
  int found=FALSE;
  char queryStr[BUFFMAXLN];
  char macHeader[ADDRMAXLN];

  /* set default values */
  vendor[0]='?';
  vendor[1]='\0';
 
  /* prepare query string */
  strncpy(macHeader, macAddress, ADDRMAXLN);
  macHeader[8]='\0';
  snprintf(queryStr, BUFFMAXLN, 
	   "select org from nicvendors where oui='%s'",
	   macHeader);

  /* send SQL query */
  if (mysql_query(&mysql, queryStr)){
     err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	     mysql_error(&mysql));
     return FALSE;
  }

  res = mysql_use_result(&mysql);
  
  /* output table name */
  row = mysql_fetch_row(res);

  /* if not found, return false */
  if(row==NULL)  found=FALSE;

  /* if found */
  else {
    strncpy(vendor, row[0], bufferLength);
    found=TRUE;
  }

  mysql_free_result(res);
  return found;
}

/******************************************
 The macAddr is already registered
******************************************/
int isMacAddrFoundInMngDb(char* macAddr){

  MYSQL_RES *res=NULL;
  MYSQL_ROW row;
  char queryStr[BUFFMAXLN];
  int ret;

  /* prepare query string */
  snprintf(queryStr, BUFFMAXLN, 
	   "select * from macaddrs "
	   " where macAddress='%s' and status!='D'",
	   macAddr);
  
  /* send SQL query */
  if (mysql_query(&mysql, queryStr)){
    err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	    mysql_error(&mysql));
    return FALSE;
  }
  
  /* store result */
  res = mysql_store_result(&mysql);

  /* output table name */
  row = mysql_fetch_row(res);
  
  /* if found, return true */
  if(row!=NULL) ret=TRUE;

  /* if not found, return false */
  else ret=FALSE;

  mysql_free_result(res);
  return ret;
}

/******************************************
 The count of registered mac address for the user
******************************************/
int countMacAddrsForUserInMngDb(char* userId, char* extraId){

  MYSQL_RES *res=NULL;
  MYSQL_ROW row;
  char queryStr[BUFFMAXLN];
  char countStr[WORDMAXLN];
  int count=10000;

  /* prepare query string */
  snprintf(queryStr, BUFFMAXLN, 
	   "select count(*) from macaddrs "
	   " where userId='%s' and extraId='%s' and status!='D'",
	   userId, extraId);

  /* send SQL query */
  if (mysql_query(&mysql, queryStr)){
    err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	    mysql_error(&mysql));
    return count;
  }

  /* store result */
  res = mysql_store_result(&mysql);

  /* output table name */
  row = mysql_fetch_row(res);

  /* if found, return values */
  if(row!=NULL){
    strncpy(countStr, row[0], WORDMAXLN);
    count=atoi(countStr);
  }

  /* free memory area */
  mysql_free_result(res);

  return count;
}

/******************************************
register mac address to the management db
******************************************/
int registMacAddrToMngDb(char* macAddr, char* deviceName, char* userId, char* extraId, char* mailAddr){

  char queryStr[BUFFMAXLN];
  int count;

   /* if alreay registered (and not deleted yet), return false */
  if(IsMacAddrFoundInMngDb(macAddr)){
    SetMessage(ExistentMacAddr);
    return FALSE;
  }
 
  /* if the user registered device more than limit, return false */
  count=CountMacAddrsForUserInMngDb(userId,extraId);
  if(count >= atoi(GetConfValue("MaxDevices"))){
    SetMessage(DeviceCountOver);
    return FALSE;
  }
 
  /* prepare query string */
  snprintf(queryStr, BUFFMAXLN, 
	   "insert into macaddrs "
	   "(macAddress, status, device, userId, extraId, "
	   " entryDate, renewDate, limitDate, mailAddress) "
	   "values ('%s','A','%s', '%s', '%s', "
	   " now(), now(), %s, '%s')",
	   macAddr, deviceName, userId, extraId, 
	   GetConfValue("LimitDate"), mailAddr);

  /* send SQL query */
  if (mysql_query(&mysql, queryStr)){
     err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	     mysql_error(&mysql));
     return FALSE;
  }

  return TRUE;
}

/*******************************************
 get next mac address from management db
 if end of list, return false
*******************************************/
int getNextMacAddrFromMngDb(char* userId, char* extraId, char* macAddress, char* deviceName, char* entryDate, char* limitDate, char* status, char* mailAddress){

  static MYSQL_RES *res=NULL;
  MYSQL_ROW row;
  char queryStr[BUFFMAXLN];

  /* set default values */
  macAddress[0]=deviceName[0]=entryDate[0]=limitDate[0]='\0';

  /* if do not get result yet */
  if(res==NULL){

    /* prepare query string */
    snprintf(queryStr, BUFFMAXLN, 
      "select macAddress, device, entryDate, limitDate, status, mailAddress "
      "from macaddrs where userId='%s' and extraId='%s' and status!='D'",
      userId, extraId);

    /* send SQL query */
    if (mysql_query(&mysql, queryStr)){
      err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	      mysql_error(&mysql));
      return FALSE;
    }
    
    /* store result */
    res = mysql_store_result(&mysql);
  }

  /* output table name */
  row = mysql_fetch_row(res);

  /* if found, return values */
  if(row!=NULL){
    strncpy(macAddress, row[0],ADDRMAXLN);
    strncpy(deviceName,row[1],WORDMAXLN);
    strncpy(entryDate,row[2],WORDMAXLN);
    strncpy(limitDate,row[3],WORDMAXLN);
    strncpy(status,row[4],WORDMAXLN);
    strncpy(mailAddress,row[5],BUFFMAXLN);
    return TRUE;
  }
  /* if not found, free memory area */
  else{
    mysql_free_result(res);
    return FALSE;
  }
}


/******************************************
 write mac modification log to the management db
******************************************/
int putMacModifyLogToMngDb(char* userId, char* extraId, char* macAddr, char modifyType){

  char queryStr[BUFFMAXLN];
 
  /* prepare query string */
  snprintf(queryStr, BUFFMAXLN, 
	   "insert into macmodify "
	   "(userId, extraId, macAddress, modifyType, modifyDate) "
	   "values ('%s', '%s', '%s', '%c', now())",
	   userId, extraId, macAddr, modifyType);

  /* send SQL query */
  if (mysql_query(&mysql, queryStr)){
     err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	     mysql_error(&mysql));
     return FALSE;
  }

  return TRUE;
}

/******************************************
 The count of mac address modification in last 24 hours
******************************************/
int countMacModifyPerDayInMngDb(char* userId, char* extraId){

  MYSQL_RES *res=NULL;
  MYSQL_ROW row;
  char queryStr[BUFFMAXLN];
  char countStr[WORDMAXLN];
  int count=10000;

  /* prepare query string */
  snprintf(queryStr, BUFFMAXLN, 
	   "select count(*) from macmodify "
	   " where userId='%s' and extraId='%s' and "
	   " modifyDate > adddate(now(), interval -1 day) ",
	   userId, extraId);

  /* send SQL query */
  if (mysql_query(&mysql, queryStr)){
    err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	    mysql_error(&mysql));
    return count;
  }

  /* store result */
  res = mysql_store_result(&mysql);

  /* output table name */
  row = mysql_fetch_row(res);

  /* if found, return values */
  if(row!=NULL){
    strncpy(countStr, row[0], WORDMAXLN);
    count=atoi(countStr);
  }

  /* free memory area */
  mysql_free_result(res);

  return count;
}

/******************************************
delete a mac address registered in the management db
 do not delete, but set the status flag to 'D'
******************************************/
int delMacAddrFromMngDb(char* macAddr){

  char queryStr[BUFFMAXLN];

  /* if the mac data was used, it is preserved for usage log */ 
  if(IsSessionFoundInMngDb(macAddr)){

    /* prepare query string */
    /* if session for the mac exists, the row is preserved */
    /* don't touch device set as status=Inactive */
    snprintf(queryStr, BUFFMAXLN, 
	     "update macaddrs set status='D',limitDate=now() "
	     " where macAddress='%s' and status='A'", macAddr);
  }else{
  
    /* prepare query string */
    /* if session does not exist, the row is removed */
    /* don't touch device set as status=Inactive */
    snprintf(queryStr, BUFFMAXLN, 
	     "delete from macaddrs "
	     " where macAddress='%s' and status='A'", macAddr);
  }

  /* send SQL query */
  if (mysql_query(&mysql, queryStr)){
    err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	    mysql_error(&mysql));
    return FALSE;
  }
  return TRUE;
}

/******************************************
renew the limit date for a mac address
 registered in the management db
******************************************/
int renewMacAddrInMngDb(char* macAddr){

  char queryStr[BUFFMAXLN];

  /* prepare query string */
  /* don't touch device set as inactive by admin */
  snprintf(queryStr, BUFFMAXLN, 
	   "update macaddrs set renewDate=now(), limitDate=%s "
	   " where status='A' and macAddress='%s'", 
	   GetConfValue("LimitDate"),macAddr);

  /* send SQL query */
  if (mysql_query(&mysql, queryStr)){
    err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	    mysql_error(&mysql));
    return FALSE;
  }
  return TRUE;
}

/******************************************
pause the usage of the device for a mac address
 registered in the management db
 (set the limitDate to now)
******************************************/
int pauseMacAddrInMngDb(char* macAddr){

  char queryStr[BUFFMAXLN];

  /* prepare query string */
  /* don't touch device set as inactive by admin */
  snprintf(queryStr, BUFFMAXLN, 
	   "update macaddrs set limitDate=now() "
	   " where status='A' and macAddress='%s' and limitDate>now()", 
	   macAddr);

  /* send SQL query */
  if (mysql_query(&mysql, queryStr)){
    err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	    mysql_error(&mysql));
    return FALSE;
  }
  return TRUE;
}

/******************************************
 is the mac address used in session log
******************************************/
int isSessionFoundInMngDb(char* macAddr){

  MYSQL_RES *res=NULL;
  MYSQL_ROW row;
  char queryStr[BUFFMAXLN];
  int ret;

  /* prepare query string */
  /* get row for addr.entry<session.open<addr.limit */
  snprintf(queryStr, BUFFMAXLN, 
	   "select * from macaddrs, sessionmd "
	   " where macaddrs.macAddress=sessionmd.macAddress and "
	   " macaddrs.macAddress='%s' and "
	   " entryDate<openTime and openTime<limitDate",
	   macAddr);
  
  /* send SQL query */
  if (mysql_query(&mysql, queryStr)){
    err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	    mysql_error(&mysql));
    return FALSE;
  }
  
  /* store result */
  res = mysql_store_result(&mysql);

  /* output table name */
  row = mysql_fetch_row(res);
  
  /* if found, return true */
  if(row!=NULL) ret=TRUE;

  /* if not found, return false */
  else ret=FALSE;

  mysql_free_result(res);
  return ret;
}

/*******************************************
 get next next usage log from management db
 if end of list, return false
*******************************************/
int getNextUsageLogFromMngDb(char* userId, char* extraId, char* macAddr, char* deviceName, char* openTime, char* gatewayName, int* weekday){
  static MYSQL_RES *res=NULL;
  MYSQL_ROW row;
  char queryStr[BUFFMAXLN];

  /* set default values */
  macAddr[0]=deviceName[0]=openTime[0]=gatewayName[0]='\0';

  /* if do not get result yet */
  if(res==NULL){

    /* prepare query string */
    /* get log where addr.entry < session.open < addr.limit */
    snprintf(queryStr, BUFFMAXLN, 
	     "select distinct macaddrs.macAddress, device, date(openTime), "
	     " gatewayName, weekday(openTime) "
	     " from macaddrs, sessionmd "
	     " where macaddrs.macAddress=sessionmd.macAddress "
	     " and entryDate < openTime and openTime < limitDate "
	     " and %s < openTime "
	     " and userId='%s' and extraId='%s' and status!='D'"
	     " order by openTime desc",
	     GetConfValue("ShowLogAfter"), userId, extraId);

    /* send SQL query */
    if (mysql_query(&mysql, queryStr)){
      err_msg("ERR at %s#%d: mysql query: %s",__FILE__,__LINE__,
	      mysql_error(&mysql));
      return FALSE;
    }
    
    /* store result */
    res = mysql_store_result(&mysql);
  }

  /* output table name */
  row = mysql_fetch_row(res);

  /* if found, return values */
  if(row!=NULL){
    strncpy(macAddr, row[0],ADDRMAXLN);
    strncpy(deviceName,row[1],WORDMAXLN);
    strncpy(openTime,row[2],WORDMAXLN);
    strncpy(gatewayName,row[3],WORDMAXLN);
    *weekday=atoi(row[4]);
    return TRUE;
  }
  /* if not found, free memory area */
  else{
    mysql_free_result(res);
    return FALSE;
  }
}

/*******************************************
 get next mail address near limit date
from management db 
*******************************************/
int getNextMailAddressFromMngDb(char* mailAddress, char* limitDate, char* device){

  static MYSQL_RES *res=NULL;
  MYSQL_ROW row;
  char queryStr[BUFFMAXLN];
  char* mailTiming=NULL;

  /* set default values */
  mailAddress[0]=limitDate[0]='\0';

  /* if do not get result yet */
  if(res==NULL){

    /* get mail send timing from conf file */
    mailTiming = GetConfValue("Mail/Timing");
    if(isNull(mailTiming)) return FALSE;

    /* prepare query string */
    /* [select mailAddress,limitDate,device from macaddrs 
        where (  date(now())=date(adddate(limitDate, interval -1 day)) 
              OR date(now())=date(adddate(limitDate, interval -7 day))  ) 
             AND status='A'] */
    snprintf(queryStr, BUFFMAXLN, 
	     "select mailAddress,limitDate,device from macaddrs "
	     " where (%s) and status='A'", mailTiming);

    /* send SQL query */
    if (mysql_query(&mysql, queryStr)){
      err_msg("ERR at %s#%d: mysql query[%s]: %s",__FILE__,__LINE__,queryStr,
	      mysql_error(&mysql));
      return FALSE;
    }
    
    /* store result */
    res = mysql_store_result(&mysql);
  }

  /* output table name */
  row = mysql_fetch_row(res);

  /* if found, return values */
  if(row!=NULL){
    strncpy(mailAddress, row[0], ADDRMAXLN);
    strncpy(limitDate, row[1], WORDMAXLN);
    strncpy(device, row[2], WORDMAXLN);
    return TRUE;
  }
  /* if not found, free memory area */
  else{
    mysql_free_result(res);
    return FALSE;
  }
}

/********************************************
 routines for debugging output
********************************************/

int InitMngDb(void){
  int ret;
  if(debug>1) err_msg("DEBUG:=>initMngDb()");
  ret=initMngDb();
  if(debug>1) err_msg("DEBUG:(%d)<=closeMngDb()",ret);
  return ret;
}
void CloseMngDb(void){
  if(debug>1) err_msg("DEBUG:=>closeMngDb()");
  closeMngDb();
  if(debug>1) err_msg("DEBUG:<=closeMngDb()");
}

int GetNextMacAddrFromMngDb(char* userId, char* extraId, char* macAddress, char* deviceName, char* entryDate, char* limitDate, char* status, char* mailAddress){
  int ret;
  if(debug>1) err_msg("DEBUG:=>getNextMacAddrFromMngDb(%s,%s)",userId,extraId);
  ret=getNextMacAddrFromMngDb(userId,extraId,macAddress,deviceName,
			      entryDate,limitDate, status,mailAddress);
  if(debug>1) err_msg("DEBUG:(%d)<=getNextMacAddrFromMngDb(,,%s,%s,%s,%s,%s,%s)",
		      ret,macAddress,deviceName,entryDate,limitDate,status,mailAddress);
  return ret;
}

int RegistMacAddrToMngDb(char* macAddr, char* deviceName, char* userId, char* extraId, char* mailAddress){
  int ret;
  if(debug>1) err_msg("DEBUG:=>registMacAddrToMngDb(%s,%s,%s,%s,%s)",
		      macAddr,deviceName,userId,extraId,mailAddress);
  ret=registMacAddrToMngDb(macAddr,deviceName,userId,extraId, mailAddress);
  if(debug>1) err_msg("DEBUG:(%d)<=registMacAddrToMngDb( )",ret);
  return ret;
}

int IsMacAddrFoundInMngDb(char* macAddr){
  int ret;
  if(debug>1) err_msg("DEBUG:=>isMacAddrFoundInMngDb(%s)", macAddr);
  ret=isMacAddrFoundInMngDb(macAddr);
  if(debug>1) err_msg("DEBUG:(%d)<=isMacAddrFoundInMngDb( )",ret);
  return ret;
}

int CountMacAddrsForUserInMngDb(char* userId, char* extraId){
  int ret;
  if(debug>1) err_msg("DEBUG:=>countMacAddrsForUserInMngDb(%s,%s)",
		      userId,extraId);
  ret=countMacAddrsForUserInMngDb(userId,extraId);
  if(debug>1) err_msg("DEBUG:(%d)<=countMacAddrsForUserInMngDb( )",ret);
  return ret;
}

int PutMacModifyLogToMngDb(char* userId, char* extraId, char* macAddr, char modifyType){
  int ret;
  if(debug>1) err_msg("DEBUG:=>putMacModifyLogToMngDb(%s,%s,%s,%c)",
		      userId,extraId,macAddr,modifyType);
  ret=putMacModifyLogToMngDb(userId,extraId,macAddr,modifyType);
  if(debug>1) err_msg("DEBUG:(%d)<=putMacModifyLogToMngDb( )",ret);
  return ret;
}

int GetNicVendorFromMngDb(char* macAddress, char* vendor, int bufferLength){
  int ret;
  if(debug>1) err_msg("DEBUG:=>getNicVendorFromMngDb(%s,,%d)",
		      macAddress, bufferLength);
  ret=getNicVendorFromMngDb(macAddress, vendor, bufferLength);
  if(debug>1) err_msg("DEBUG:(%d)<=getNicVendorFromMngDb(%s)",ret, vendor);
  return ret;
}


int CountMacModifyPerDayInMngDb(char* userId, char* extraId){
  int ret;
  if(debug>1) err_msg("DEBUG:=>countMacModifyPerDayInMngDb(%s,%s)",
		      userId,extraId);
  ret=countMacModifyPerDayInMngDb(userId,extraId);
  if(debug>1) err_msg("DEBUG:(%d)<=countMacModifyPerDayInMngDb( )",ret);
  return ret;
}

int DelMacAddrFromMngDb(char* macAddr){
  int ret;
  if(debug>1) err_msg("DEBUG:=>delMacAddrFromMngDb(%s)",macAddr);
  ret=delMacAddrFromMngDb(macAddr);
  if(debug>1) err_msg("DEBUG:(%d)<=delMacAddrFromMngDb( )",ret);
  return ret;
}

int RenewMacAddrInMngDb(char* macAddr){
  int ret;
  if(debug>1) err_msg("DEBUG:=>renewMacAddrInMngDb(%s)",macAddr);
  ret=renewMacAddrInMngDb(macAddr);
  if(debug>1) err_msg("DEBUG:(%d)<=renewMacAddrinMngDb( )",ret);
  return ret;
}

int PauseMacAddrInMngDb(char* macAddr){
  int ret;
  if(debug>1) err_msg("DEBUG:=>pauseMacAddrInMngDb(%s)",macAddr);
  ret=pauseMacAddrInMngDb(macAddr);
  if(debug>1) err_msg("DEBUG:(%d)<=pauseMacAddrinMngDb( )",ret);
  return ret;
}

int IsSessionFoundInMngDb(char* macAddr){
  int ret;
  if(debug>1) err_msg("DEBUG:=>isSessionFoundInMngDb(%s)",macAddr);
  ret=isSessionFoundInMngDb(macAddr);
  if(debug>1) err_msg("DEBUG:(%d)<=isSessionFoundInMngDb( )",ret);
  return ret;
}

int GetNextUsageLogFromMngDb(char* userId, char* extraId, char* macAddr, char* deviceName, char* openTime, char* gatewayName, int* weekday){
  int ret;
  if(debug>1) err_msg("DEBUG:=>getNextUsageLogFromMngDb(%s,%s)",userId,extraId);
  ret=getNextUsageLogFromMngDb(userId,extraId,macAddr,deviceName,
			      openTime,gatewayName,weekday);
  if(debug>1) err_msg("DEBUG:(%d)<=getNextUsageLogFromMngDb(,,%s,%s,%s,%s,%d)",
		      ret,macAddr,deviceName,openTime,gatewayName,*weekday);
  return ret;
}

int GetNextMailAddressFromMngDb(char* mailAddress, char* limitDate, char*device){
  int ret;
  if(debug>1) err_msg("DEBUG:=>getnextMailAddressFromMngDb( )");
  ret=getNextMailAddressFromMngDb(mailAddress, limitDate, device);
  if(debug>1) err_msg("DEBUG:(%d)<=getNextMailAddressFromMngDb(%s,%s,%s)",
		      ret,mailAddress,limitDate,device);
  return ret;
}
