/**************************************************
Opengate mac addr auth program
 module for Communication through CGI 

Copyright (C) 2011 Opengate Project Team
Written by Yoshiaki Watanabe

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Email: watanaby@is.saga-u.ac.jp

Programmed by Yoshiaki WATANABE
**************************************************/

#include	"opengatemmng.h"

/* convert one-char-hex "a" to one-number 0Xa */ 
#define hex2num(x)  ((x)>='A' ? ((x) & 0XDF) - 'A' +10 : ((x) - '0'))

void split(char content[], char *name[], char *value[], char *next[]);
void decode(char *string);

char macAddress[ADDRMAXLN]="";

/********************************************/
/* get Post data from the client  */
/********************************************/
int getPostData(char *content, int contentMaxLength)
{
  int contentLen;

  /* get content sent from web input */
  if(isNull(getenv("CONTENT_LENGTH"))) return FALSE;
  contentLen=atoi(getenv("CONTENT_LENGTH"));
  contentLen++; /* for terminate ch */

  /* if larger than buffer, cut off */
  if(contentLen > contentMaxLength) contentLen=contentMaxLength;
  if(fgets(content, contentLen, stdin) == NULL){
    content[0]='\0';
  }

  return TRUE;
}

/********************************************/
/* get language form query string (in url lang=ja)  */
/********************************************/
int getLangFromQueryString(char* language){

  int found=FALSE;
  char queryStr[BUFFMAXLN];
  char *name[1];
  char *value[1];
  char *next[1];
  char *ptr;

  /* get default language at the top of lang list */
  sscanf(GetConfValue("HtmlLangs"), "%s", language);

  /* if no string, return default */
  if(isNull(getenv("QUERY_STRING"))) return FALSE;

  /* get html access parameter string */
  strncpy(queryStr, getenv("QUERY_STRING"), BUFFMAXLN);

  /* split language in string [....&lang=ja&.....] */
  ptr=queryStr;
  while(ptr!=NULL){
    split(ptr, name, value, next);
    if(strstr(name[0], "lang")!=NULL){
      strncpy(language, value[0], WORDMAXLN);
      found=TRUE;
    }
    ptr=next[0];
  }
  return found;

}

/********************************************/
/* get mac address form query string (in url lang=ja)  */
/********************************************/
int getMacAddrFromQueryString(char* macAddress){

  int found=FALSE;
  char queryStr[BUFFMAXLN];
  char *name[1];
  char *value[1];
  char *next[1];
  char *ptr;

  /* set default */
  macAddress[0]='\0';

  /* if no string returns, return default */
  if(isNull(getenv("QUERY_STRING"))) return FALSE;

  /* get html access parameter string */
  strncpy(queryStr, getenv("QUERY_STRING"), BUFFMAXLN);

  /* split in string [....&macaddr=xxxxx&.....] */
  ptr=queryStr;
  while(ptr!=NULL){
    split(ptr, name, value, next);
    if(strstr(name[0], "macaddr")!=NULL){
      strncpy(macAddress, value[0], WORDMAXLN);
      found=TRUE;
    }
    ptr=next[0];
  }
  decode(macAddress);
  return found;

}

/********************************************/
/* get userid from environment variable     */
/********************************************/
int getUserIdFromEnv(char *userid){

  int ret=FALSE;
  char* pEnv=NULL;

  /* if shibboleth or httpbasic, get uid from environment var */
  /* shibboleth */
  if(strcmp(GetConfValue("AuthServer/Protocol"), "shibboleth")==0){
    pEnv=getenvEx(GetConfValue("AuthServer/UidAttribute"),TRUE,TRUE); 
    if(!isNull(pEnv)){
      strncpy(userid, pEnv, USERMAXLN);

      /* if idp string can be get from env variable, concatinate it as uid@idp */
      pEnv=getenvEx(GetConfValue("AuthServer/OrgAttribute"),TRUE,TRUE);
      if(!isNull(pEnv)){
	strncat(userid, GetConfValue("UserIdSeparator"), USERMAXLN);
	strncat(userid, pEnv, USERMAXLN);
      } 
      ret=TRUE;
    }else{
      err_msg("ERR at %s#%d: Cannot get user info from shibboleth",__FILE__,__LINE__);
      ret=FALSE;
      PutMessageToClient("Cannot get user info from shibboleth<br>Check shibboleth setting in .htaccess and other");
      exit(0);
    }    
  }

  /* httpbasic */
  else if(strcmp(GetConfValue("AuthServer/Protocol"), "httpbasic")==0){
    if(!isNull(getenv("REMOTE_USER"))){
      strncpy(userid,getenv("REMOTE_USER"),USERMAXLN);
      ret=TRUE;
    }else{
      err_msg("ERR at %s#%d: Cannot get user info from httpbasic",__FILE__,__LINE__);
      ret=FALSE;
      PutMessageToClient("Cannot get user info from http basic<br>Check http basic setting in .htaccess and other");
      exit(0);
    }
  }
  return ret;
}

/*******************************
get userid and password from post string
*******************************/
int getUserIdFromPostData(char* requestStr, char* userid, char* password){

  char *name[1];
  char *value[1];
  char *next[1];
  char *ptr;
  int ret=FALSE;
  char content[BUFFMAXLN];

  /* if null string, return */
  if(isNull(requestStr)) return FALSE;

  /* copy it to work area */
  strncpy(content, requestStr, BUFFMAXLN);

  /* split request item and execute the request */
  ptr=content;
  while(ptr!=NULL){

    /* pick up next item */
    split(ptr, name, value, next);
    
    /* copy to var */
    if(strcmp(name[0], "userid")==0){
      strncpy(userid, value[0], USERMAXLN);
      decode(userid);
      ret=TRUE;
    }
    else if(strcmp(name[0], "password")==0){
      strncpy(password, value[0], USERMAXLN);
      decode(password);
    }
    
    /* shift pointer to next item */
    ptr=next[0];
  }
  return ret;
}

/********************************************/
/* analyze request for checking    */
/********************************************/
int analyzeCheckRequest(char *content, int* status, char* macAddress)
{
  char *name[1];
  char *value[1];
  char *next[1];
  char *ptr;

  /* set default */
  *status=NONE;
  *macAddress='\0';

  /* split request item and execute the request */
  ptr=content;
  while(!isNull(ptr)){

    /* pick up next item */
    split(ptr, name, value, next);

    /* if item =status */
    if(strcmp(name[0], "status")==0){
      if(strcmp(value[0], "open")==0) *status=OPEN;
      if(strcmp(value[0], "close")==0) *status=CLOSE;
    }

    /* if item = macaddr */
    else if(strcmp(name[0], "macaddr")==0){
      strncpy(macAddress, value[0], ADDRMAXLN);
      decode(macAddress);
    }
    
    /* shift pointer to next item */
    ptr=next[0];
  }

  return TRUE;
}

/********************************************/
/* analyze request and execute request for registering   */
/********************************************/
int analyzeRegisterRequest(char *content, char* macAddress, char* deviceName, char* mailAddress)
{
  char *name[1];
  char *value[1];
  char *next[1];
  char *ptr;

  /* set default */
  *macAddress='\0';
  *deviceName='\0';
  *mailAddress='\0';

  /* split request item and execute the request */
  ptr=content;
  while(!isNull(ptr)){

    /* pick up next item */
    split(ptr, name, value, next);

    /* if item = macaddr */
    if(strcmp(name[0], "macaddr")==0){
      strncpy(macAddress, value[0], ADDRMAXLN);
      decode(macAddress);
    }
    else if(strcmp(name[0], "device")==0){
      strncpy(deviceName, value[0], WORDMAXLN);
      decode(deviceName);
    }
    else if(strcmp(name[0], "mailaddr")==0){
      strncpy(mailAddress, value[0], BUFFMAXLN);
      decode(mailAddress);
    }
     
    /* shift pointer to next item */
    ptr=next[0];
  }

  /* if no data is inputed, return */
  if(*deviceName=='\0' && *mailAddress=='\0') return FALSE;

  /* if illegal device name, return */
  if(*deviceName=='\0'){
    SetMessage(EmptyDeviceName); 
    return FALSE;
  }
  if(!IsSafeString(deviceName, WORDMAXLN)){
    SetMessage(IllegalCharInDevice);
    return FALSE;
  }

  /* if illegal mail address, return */
  if(!IsSafeString(mailAddress, BUFFMAXLN)){
    SetMessage(IllegalCharInMailAddr);
    return FALSE;
  }

  /* if illegal mac address, return */
  if(!ConvertMacAddr(macAddress)) return FALSE;

  /* if already registered in db, return */
  if(IsMacAddrFoundInMngDb(macAddress)){
    SetMessage(ExistentMacAddr);
    return FALSE;
  }
  
  return TRUE;
}
  
/******************************
 convert mac address to regular format
  [01:23:45:67:89:0a]
******************************/
int convertMacAddr(char* macAddr){

  int m[6]; /* mac address sequence */
  int i;

  /* normal form is hex:hex:hex:hex:hex:hex */
  /*             or hex-hex-hex-hex-hex-hex */
  /* and hex<256 */
  if((sscanf(macAddr, "%x:%x:%x:%x:%x:%x", 
	     &m[0],&m[1],&m[2],&m[3],&m[4],&m[5])!=6)
     && (sscanf(macAddr, "%x-%x-%x-%x-%x-%x", 
		&m[0],&m[1],&m[2],&m[3],&m[4],&m[5])!=6)){
    SetMessage(IllegalMacAddrForm);
    return FALSE;
  }
  for(i=0;i<6;i++){
    if(m[i]>255){
      SetMessage(IllegalMacAddr);
      return FALSE;
    }
  }

  /* format to [01:23:45:67:89:0a] */
  snprintf(macAddr, ADDRMAXLN, "%02x:%02x:%02x:%02x:%02x:%02x",
	   m[0],m[1],m[2],m[3],m[4],m[5]);

  return TRUE;
}

/******************************
check safe characters string 
******************************/
int isSafeString(char* str, int length){
  char allowableChar[]=" !#$&*+,-./:=?[]^_{|}@";
  int i;
  char ch;

  /* scan all char in str */
  for(i=0; i<length; i++){
    ch=str[i];
    
    /* if reached to the end of string, return true */
    if(ch=='\0') break;

    /* if alpha or numeric is found, goto next char */
    if(isalnum(ch)) continue;

    /* if control or non ascii is found, return false */
    if( (unsigned)ch < 0x20|| 0x7f < (unsigned)ch) return FALSE;
    
    /* if one of above list is found, goto next char */
    if(strchr(allowableChar, ch)==NULL) return FALSE;
  }

  return TRUE;
}

/*********************************************/
/* put auth request page to client            */
/*********************************************/
int putAuthRequestPageToClient(char *language, char* cgiName)
{
  char authdoc[BUFFMAXLN];
  FILE *fp;
  char buff[BUFFMAXLN];

  /* make read in path to the retry document */
  snprintf(authdoc, BUFFMAXLN, "%s%s/%s/%s",GetConfValue("DocumentRoot"),
	  GetConfValue("OpengateDir"),language,GetConfValue("AuthDoc"));

  /* replace keyword and send out the file */
  printf("Content-type: text/html\r\n\r\n");

  if((fp=fopen(authdoc, "r"))==NULL){
    err_msg("ERR at %s#%d: cannot find file %s",__FILE__,__LINE__, authdoc);
    return FALSE;
  }

  while(fgets(buff,BUFFMAXLN,fp)!=NULL){
    HtmlReplace(buff, "%%CGINAME%%", cgiName);
    if(strstr(buff, "%%ERRORLIST%%")!=NULL){
      InsertMessageToPage(language);
    }else{
      printf("%s",buff);    
    }
  }
  fclose(fp);
  return TRUE;
}

/*********************************************/
/* deny message to the client            */
/*********************************************/
int putDenyToClient(char *language){
  char denydoc[BUFFMAXLN];
  FILE *fp;
  char buff[BUFFMAXLN];

  /* make read in path to the retry document */
  snprintf(denydoc, BUFFMAXLN, "%s%s/%s/%s",GetConfValue("DocumentRoot"),
	  GetConfValue("OpengateDir"),language,GetConfValue("DenyDoc"));

  /* replace keyword and send out the file */
  printf("Content-type: text/html\r\n\r\n");

  if((fp=fopen(denydoc, "r"))==NULL){
    err_msg("ERR at %s#%d: cannot find file %s",__FILE__,__LINE__, denydoc);
    return FALSE;
  }

  while(fgets(buff,BUFFMAXLN,fp)!=NULL){
    if(strstr(buff, "%%ERRORLIST%%")!=NULL){
      InsertMessageToPage(language);
    }else{
      printf("%s",buff);
    }
  }
  fclose(fp);
  return TRUE;
}

/*********************************************/
/* put response to client for check request   */
/*********************************************/
int putCheckPageToClient(char *language, char* userId, char* extraId)
{
  char responsedoc[BUFFMAXLN];
  FILE *fp;
  char buff[BUFFMAXLN];
  char cookie[SIDMAXLN];
  char* chkCgi=GetConfValue("CheckCgi");
  char* regCgi=GetConfValue("RegisterCgi");
  char* timeout=GetConfValue("OpenTimeout");

  /* make read in path to the document */
  snprintf(responsedoc, BUFFMAXLN, "%s%s/%s/%s",GetConfValue("DocumentRoot"),
	  GetConfValue("OpengateDir"),language,GetConfValue("CheckDoc"));

  /* send header */
  printf("Content-type: text/html\r\n");

  /* if no cookie, make, send, and save cookie */
  if(!GetHttpCookie(cookie,GetConfValue("AuthAdminCookie"))){
    CreateCookie(cookie);
    printf("Set-Cookie: %s=%s;path=/;\r\n", GetConfValue("AuthAdminCookie"), cookie);
    SaveCookieToWorkDb(cookie,userId, extraId, ADMINUSER);
  }

  /* end of http header */
  printf("\r\n");

  /* replace keyword and send out the file */
  if((fp=fopen(responsedoc, "r"))==NULL){
    err_msg("ERR at %s#%d: cannot find file %s",__FILE__,__LINE__, responsedoc);
    return FALSE;
  }


  while(fgets(buff,BUFFMAXLN,fp)!=NULL){
    HtmlReplace(buff, "%%CHECKCGI%%", chkCgi);
    HtmlReplace(buff, "%%REGISTERCGI%%", regCgi);
    HtmlReplace(buff, "%%TIMEOUT%%", timeout);
    HtmlReplace(buff, "%%USERID%%", userId);

    if(strstr(buff, "%%MACCHECKLIST%%")!=NULL){
      PutMacCheckListToClient();
    }
    else if(strstr(buff, "%%ERRORLIST%%")!=NULL){
      InsertMessageToPage(language);
    }

    else{
      printf("%s",buff);    
    }
  }
  fclose(fp);
  return TRUE;
}
/*********************************************/
/* put response to client for register request */
/*********************************************/
int putRegisterPageToClient(char *language, char* macAddress, char* deviceName, char* mailAddress, char* userId, char* extraId)
{
  char responsedoc[BUFFMAXLN];
  FILE *fp;
  char buff[BUFFMAXLN];
  char cookie[SIDMAXLN];
  char* chkCgi=GetConfValue("CheckCgi");
  char* regCgi=GetConfValue("RegisterCgi");

  /* make read in path to the document */
  snprintf(responsedoc, BUFFMAXLN, "%s%s/%s/%s",GetConfValue("DocumentRoot"),
	  GetConfValue("OpengateDir"),language,GetConfValue("RegisterDoc"));

  /* send header */
  printf("Content-type: text/html\r\n");

  /* if no cookie, make, send, and save cookie */
  if(!GetHttpCookie(cookie, GetConfValue("AuthUserCookie"))){
    CreateCookie(cookie);
    printf("Set-Cookie: %s=%s;path=/;\r\n", GetConfValue("AuthUserCookie"), cookie);
    SaveCookieToWorkDb(cookie, userId, extraId, NORMALUSER);
    SaveMailDefalutForCookieToWorkDb(cookie, mailAddress);
  }

  /* end of http header */
  printf("\r\n");

  if((fp=fopen(responsedoc, "r"))==NULL){
    err_msg("ERR at %s#%d: cannot find file %s",__FILE__,__LINE__, responsedoc);
    return FALSE;
  }

  while(fgets(buff,BUFFMAXLN,fp)!=NULL){
    HtmlReplace(buff, "%%CHECKCGI%%", chkCgi);
    HtmlReplace(buff, "%%REGISTERCGI%%", regCgi);
    HtmlReplace(buff, "%%MACADDR%%", macAddress);
    HtmlReplace(buff, "%%DEVICE%%", deviceName);
    HtmlReplace(buff, "%%MAILADDR%%", mailAddress);
    HtmlReplace(buff, "%%USERID%%", userId);

    if(strstr(buff, "%%MACREGLIST%%")!=NULL){
      PutMacRegListToClient(userId, extraId);
    }
    else if(strstr(buff, "%%ERRORLIST%%")!=NULL){
      InsertMessageToPage(language);
    }
    else{
      printf("%s",buff);    
    }
  }
  fclose(fp);
  return TRUE;
}

/*********************************************/
/* put mac list in mac check table to the client */
/*********************************************/
void putMacCheckListToClient(void){

  char macAddress[ADDRMAXLN];
  char vendor[WORDMAXLN];
  char ipv4[ADDRMAXLN];
  char ipv6[ADDRMAXLN];
  int firstRow=TRUE;
  int inUse=FALSE;
  int foundOnDb=FALSE;

  /* get mac list from db and insert into html table */
  while(GetNextRowInMacCheckTable(macAddress, ipv4, ipv6)){

    /* get nic vendor from management db */
    GetNicVendorFromMngDb(macAddress, vendor, WORDMAXLN);

    /* is the terminal in use */
    if(IsSessionFoundInSessionTable(macAddress) ||
       IsActiveSessionFoundInOpengateSessionTable(macAddress)) inUse=TRUE;
    else inUse = FALSE;
    
    /* is the terminal's MAC is registered in DB */
    foundOnDb = IsMacAddrFoundInMngDb(macAddress);

    /* print out table row */
    /* the row is colored, if it is a candidate for registering */
    /* the terminals inUse/foundInDb may be not the candidate */
    if(inUse || foundOnDb){
      printf("<tr align=middle>\n");
    }else{
      printf("<tr style='background-color: rgb(255,255,204);' align=middle>\n");
    }

    /* check radio button in first row */ 
    if(firstRow){
      printf("<td><input type='radio' name='macaddr' value='%s' checked></td>\n", macAddress);
      firstRow=FALSE;
    }else{
      printf("<td><input type='radio' name='macaddr' value='%s'></td>\n", macAddress);
    }

    /* show macAddress, vendor, ipv4,ipv6 */
    printf("<td>%s</td>\n", macAddress);
    printf("<td>%s</td>\n",vendor);
    printf("<td>%s</td>\n",ipv4);
    printf("<td>%s</td>\n",ipv6);

    /* show flags for inUse/foundInDb */
    if(inUse) printf("<td>*</td>\n");
    else  printf("<td><br></td>\n");
    if(foundOnDb) printf("<td>*</td>\n");
    else  printf("<td><br></td>\n");
    printf("</tr>\n");
  }
}

/*********************************************/
/* put mac regsitered list the client            */
/*********************************************/
void putMacRegListToClient(char* userId, char* extraId){

  char deviceName[WORDMAXLN]="";
  char entryDate[WORDMAXLN]="";
  char limitDate[WORDMAXLN]="";
  char status[WORDMAXLN]="";
  char macAddress[ADDRMAXLN]="";
  char mailAddress[BUFFMAXLN]="";

  /* get registered mac list form db and insert */
  while(GetNextMacAddrFromMngDb(userId,extraId,macAddress,deviceName,
				entryDate,limitDate,status,mailAddress)){


    printf("<tr align=middle>\n");
    printf("<td>%s</td>\n", macAddress);
    printf("<td>%s</td>\n",deviceName);
    printf("<td>%s</td>\n",entryDate);
    printf("<td>%s</td>\n",limitDate);
    printf("<td>%s</td>\n",status);
    printf("<td>%s</td>\n",mailAddress);
    printf("</tr>\n");
  }
}

/*********************************************/
/* put some message to the client            */
/*********************************************/
void putMessageToClient(char *message)
{
  printf("Content-type: text/html\r\n\r\n");
  printf("<HTML><HEAD><TITLE>OpengateMsg</TITLE></HEAD> \r\n");
  printf("<BODY>\r\n");
  printf("%s\r\n",     message);
  printf("</BODY></HTML> \r\n\r\n");

}

/************************************/
/* split value for indicated name   */
/* in content  "name=value&..."     */
/************************************/
void split(char content[], char *name[], char *value[], char *next[])
{
  char *pstr;
  
  /* set default */
  name[0]=content;
  value[0]=content+strlen(content);
  next[0]=value[0];

  /* set name end */
  if((pstr=strchr(name[0],(int)'='))==NULL){
    next[0]=NULL;
    return;
  }
  *pstr='\0';
  
  /* set value start */
  pstr++;
  value[0]=pstr;
  
  /* set value end */
  if((pstr=strchr(value[0],'&'))==NULL){
    next[0]=NULL;
    return;
  }
  *pstr='\0';

  /* set next start */
  pstr++;
  next[0]=pstr;

  return;
}

/**********************************/
/* decode text coding in web post */
/**********************************/
void decode(char *string)
{
  char *pcheck, *pinsert;

  pcheck=pinsert=string;
  while(*pcheck != '\0'){
    if(*pcheck == '+'){
      *pinsert = ' ';
    }else if(*pcheck == '%'){
      *pinsert=(char)(hex2num(*(pcheck+1))*16 + hex2num(*(pcheck+2)));
      pcheck+=2;
    }else{
      *pinsert=*pcheck;
    }
    pcheck++;
    pinsert++;
  }
  *pinsert=*pcheck;
}


/*****************************************************/
/* replace beforeStr to afterStr in string in buff   */
/*****************************************************/
int htmlReplace(char* buff,char *beforeStr,char *afterStr)
{
  char *pBuff , *pNext;
  char tempBuff[BUFFMAXLN];
  
  if(buff==NULL) return 1;

  strncpy(tempBuff, buff, BUFFMAXLN);
  strncpy(buff,"",BUFFMAXLN);
  
  for(pBuff = tempBuff;
      (pNext=StrSplit(pBuff, beforeStr)) != NULL;
      pBuff = pNext){
    strncat(buff,pBuff,BUFFMAXLN);
    strncat(buff,afterStr,BUFFMAXLN);
  }
  strncat(buff,pBuff,BUFFMAXLN);
  
  return 0;
}

/*****************************************************/
/* split a str at delimStr and return the point      */
/*****************************************************/
char* strSplit(char* str,const char* delimStr)
{
    char* delimPoint = strstr(str,delimStr);
    const size_t delimLen = strlen(delimStr);

    if(delimPoint == NULL) return NULL;
    else{
        *delimPoint = '\0';
        delimPoint += delimLen; 
    }
    return delimPoint;
}


/**********************/
/* get HTTP-Cookie    */
/**********************/
  /* cookie string examples 
  "OpengateMmng=de..ac1&Userid=user1"
  "OpengateMmng=de..ac1&Userid=user1; xxx=..; yyy=.."
  "xxx=..; yyy=..; OpengateMmng=de..ac1&Userid=user1"
  */
int getHttpCookie(char *cookie, char* cookieName){
  char content[BUFFMAXLN];
  char *name[1];
  char *value[1];
  char *next[1];
  char *ptr=NULL;
  char *ptrNext=NULL;

  /* reset buffer */
  cookie[0]='\0';

 /* if exist cookie, copy it to work area */
  if(isNull(getenv("HTTP_COOKIE"))) return FALSE;
  strncpy(content, getenv("HTTP_COOKIE"), BUFFMAXLN);
  ptr=content;

  /* search 'OpengateMmng' cookie string (terminated by ; or \0) */
  while(ptr!=NULL){
    if((ptrNext=strstr(ptr, "; "))==NULL) break;          /* search "; " */
    *ptrNext='\0';                               /* overwrite string end */
    ptrNext++;                                 /* pointer to next string */
    while(!isNull(ptrNext)&&*ptrNext==' ') ptrNext++;     /* skip spaces */
    if(strstr(ptr, cookieName)==ptr) break;          /* exit at matching */
    ptr=ptrNext;                                    /* check next string */
  }

  /* get valuses of cookie from "OpengateMmng=de..ac1" */
  while(ptr!=NULL){
    split(ptr, name, value, next);

    if(strstr(name[0], cookieName)!=NULL){
      strncpy(cookie, value[0], SIDMAXLN);
    }
    ptr=next[0];
  }
  
  if(isNull(cookie)) return FALSE;
  else return TRUE;
}

/*************************************
compare received cookie to previously saved one
*************************************/
int isCorrectCookie(char* cookie, int userType){
  char userId[USERMAXLN];
  char extraId[USERMAXLN];

  /* compare http received cookie and DB readin cookie */
  switch(userType){
  case NORMALUSER:
    GetHttpCookie(cookie, GetConfValue("AuthUserCookie"));
    if(IsCookieFoundInWorkDb(cookie,userId,extraId,NORMALUSER)) return TRUE;
    else return FALSE;
  case ADMINUSER:
    GetHttpCookie(cookie, GetConfValue("AuthAdminCookie"));
    if(IsCookieFoundInWorkDb(cookie,userId,extraId,ADMINUSER)) return TRUE;
    else return FALSE;
  }
  return FALSE;
}

/********************************************/
/* analyze update request and execute request    */
/********************************************/
int analyzeUpdateRequestAndExecute(char *requestStr, char* userId, char* extraId)
{
  char *name[1];
  char *value[1];
  char *next[1];
  char *ptr;
  char macAddr[ADDRMAXLN];
  int modified=FALSE;    /* database modification is executed */
  int ret;
  char content[BUFFMAXLN];

  /* if null string, return */
  if(isNull(requestStr)) return FALSE;

  /* copy request string to work area */
  strncpy(content, requestStr, BUFFMAXLN);

  /* split request item and execute the request */
  ptr=content;
  while(ptr!=NULL){

    /* pick up next item */
    split(ptr, name, value, next);
    
    /* if item=delete, execute delete */
    if(strcmp(name[0], "delete")==0){
      strncpy(macAddr, value[0], ADDRMAXLN);
      decode(macAddr);
      if(ConvertMacAddr(macAddr)){
	ret=DelMacAddrFromMngDb(macAddr);
	if(ret){
	  modified=TRUE;
	  PutMacModifyLogToMngDb(userId, extraId, macAddr, 'D');
	}
      }
    }

    /* if item = extend, execute extend */
    else if(strcmp(name[0], "extend")==0){
      strncpy(macAddr, value[0], ADDRMAXLN);
      decode(macAddr);
      if(ConvertMacAddr(macAddr)){
	ret=RenewMacAddrInMngDb(macAddr);
	if(ret){
	  modified=TRUE;
	  PutMacModifyLogToMngDb(userId, extraId, macAddr, 'E');
	}
      }
    }

    /* if item = pause, execute pause */
    else if(strcmp(name[0], "pause")==0){
      strncpy(macAddr, value[0], ADDRMAXLN);
      decode(macAddr);
      if(ConvertMacAddr(macAddr)){
	ret=PauseMacAddrInMngDb(macAddr);
	if(ret){
	  modified=TRUE;
	  PutMacModifyLogToMngDb(userId, extraId, macAddr, 'P');
	}
      }
    }
    
    /* shift pointer to next item */
    ptr=next[0];
  }

  return modified;
}


/*********************************************/
/* put response to client            */
/*********************************************/
int putUpdatePageToClient(char *language, char* userId, char* extraId)
{
  char responsedoc[BUFFMAXLN];
  FILE *fp;
  char buff[BUFFMAXLN];
  char cookie[SIDMAXLN];

  /* make read in path to the retry document */
  snprintf(responsedoc, BUFFMAXLN, "%s%s/%s/%s",GetConfValue("DocumentRoot"),
	  GetConfValue("OpengateDir"),language,GetConfValue("UpdateDoc"));

  /* send header */
  printf("Content-type: text/html\r\n");

  /* if no cookie, make, send, and save cookie */
  if(!GetHttpCookie(cookie, GetConfValue("AuthUserCookie"))){
    CreateCookie(cookie);
    printf("Set-Cookie: %s=%s;path=/;\r\n", GetConfValue("AuthUserCookie"), cookie);
    SaveCookieToWorkDb(cookie,userId, extraId, NORMALUSER);
  }

  /* end of http header */
  printf("\r\n");

  if((fp=fopen(responsedoc, "r"))==NULL){
    err_msg("ERR at %s#%d: cannot find file %s",__FILE__,__LINE__, responsedoc);
    return FALSE;
  }

  while(fgets(buff,BUFFMAXLN,fp)!=NULL){
    HtmlReplace(buff, "%%CGINAME%%", GetConfValue("UpdateCgi"));
    HtmlReplace(buff, "%%USERID%%", userId);

    if(strstr(buff, "%%MACLIST%%")!=NULL){
      PutMacListToClient(userId,extraId);
    }
    else if(strstr(buff, "%%USAGELOG%%")!=NULL){
      PutUsageLogToClient(userId, extraId, language);
    }
    else if(strstr(buff, "%%ERRORLIST%%")!=NULL){
      InsertMessageToPage(language);
    }
    else{
      printf("%s",buff);    
    }
  }
  fclose(fp);
  return TRUE;
}

/*********************************************/
/* put mac list the client            */
/*********************************************/
void putMacListToClient(char* userId, char* extraId){

  char deviceName[WORDMAXLN];
  char entryDate[WORDMAXLN];
  char limitDate[WORDMAXLN];
  char status[WORDMAXLN];
  char macAddr[ADDRMAXLN];
  char mailAddress[BUFFMAXLN];

  /* make rows for deletion and extending */
  /* get registered mac list form db and insert */
  while(GetNextMacAddrFromMngDb(userId,extraId,macAddr,deviceName,
				entryDate,limitDate,status,mailAddress)){
    printf("<tr align=middle>\n");
    if(status[0]=='A'){
      printf("<td><input type=checkbox name=extend value=%s></td>\n",macAddr);
      printf("<td><input type=checkbox name=pause value=%s></td>\n",macAddr);
      printf("<td><input type=checkbox name=delete value=%s></td>\n",macAddr);
    }else{
      printf("<td>-</td>\n");
      printf("<td>-</td>\n");
      printf("<td>-</td>\n");
    }
    printf("<td>%s</td>\n", macAddr);
    printf("<td>%s</td>\n",deviceName);
    printf("<td>%s</td>\n",entryDate);
    printf("<td>%s</td>\n",limitDate);
    printf("<td>%s</td>\n",status);
    printf("</tr>\n");
  }
}

/*********************************************/
/* put usage log to the client            */
/*********************************************/
void putUsageLogToClient(char *userId, char* extraId, char* language){

  char macAddr[ADDRMAXLN];
  char deviceName[WORDMAXLN];
  char openTime[WORDMAXLN];
  char gatewayName[WORDMAXLN];
  int weekday;
  char str[WORDMAXLN];
  char* p;

  /* get usage log form db and insert as html table row */
  while(GetNextUsageLogFromMngDb(userId,extraId,macAddr,deviceName,
				 openTime,gatewayName,&weekday)){

    /* split hostname only */
    if((p=strchr(gatewayName,'.'))!=NULL) *p='\0';

    /* put out table row */
    printf("<tr align=middle>\n");
    printf("<td>%s</td>\n", macAddr);
    printf("<td>%s</td>\n",deviceName);
    printf("<td>%s</td>\n",openTime);
    printf("<td>%s</td>\n",WeekdayStr(weekday, str, language));
    printf("<td>%s</td>\n",gatewayName);
    printf("</tr>\n");
  }
}


/*******************************/
/*******************************/

int GetPostData(char *content, int contentMaxLength){
  int ret;
  if(debug>1) err_msg("DEBUG:=>getPostData(%d)", contentMaxLength);
  ret=getPostData(content, contentMaxLength);
  if(debug>1) err_msg("DEBUG:%d<=getPostData(%s)",ret, content);
  return ret;
}

int GetLangFromQueryString(char* language){
  int ret;
  if(debug>1) err_msg("DEBUG:=>getLangFromQueryString( )");
  ret=getLangFromQueryString(language);
  if(debug>1) err_msg("DEBUG:%d<=getLangFromQueryString(%s)",ret,language);
  return ret;
}
int GetMacAddrFromQueryString(char* macAddress){
  int ret;
  if(debug>1) err_msg("DEBUG:=>getMacAddrFromQueryString( )");
  ret=getMacAddrFromQueryString(macAddress);
  if(debug>1) err_msg("DEBUG:%d<=getMacAddrFromQueryString(%s)",ret,macAddress);
  return ret;
}

int AnalyzeCheckRequest(char *content, int* status, char* macAddress){
  int ret;
  if(debug>1) err_msg("DEBUG:=>analyzeCheckRequest(%s)", content);
  ret=analyzeCheckRequest(content, status, macAddress);
  if(debug>1) err_msg("DEBUG:%d<=analyzeCheckRequest(%d,%s)",ret,*status, macAddress);
  return ret;
}

int AnalyzeRegisterRequest(char *content, char* macAddress, char* deviceName, char* mailAddress){
  int ret;
  if(debug>1) err_msg("DEBUG:=>analyzeRegisterRequestAndExecute(%s)", content);
  ret=analyzeRegisterRequest(content, macAddress, deviceName, mailAddress);
  if(debug>1) err_msg("DEBUG:%d<=analyzeRegisterRequestAndExecute(%s,%s,%s)",ret, macAddress, deviceName, mailAddress);
  return ret;
}

int PutDenyToClient(char *language){
  int ret;
  if(debug>1) err_msg("DEBUG:=>putDenyToClient(%s)",language);
  ret=putDenyToClient(language);
  if(debug>1) err_msg("DEBUG:(%d)<=putDenyToClient( )",ret);
  return ret;
}

int PutCheckPageToClient(char *language, char* userId, char* extraId){
  int ret;
  if(debug>1) err_msg("DEBUG:=>putCheckPageToClient(%s,%s,%s)", 
		      language,userId,extraId);
  ret=putCheckPageToClient(language,userId,extraId);
  if(debug>1) err_msg("DEBUG:(%d)<=putCheckPageToClient( )",ret);
  return ret;
}

int PutRegisterPageToClient(char *language, char* macAddress, char* deviceName, char* mailAddress, char* userId, char* extraId){
  int ret;
  if(debug>1) err_msg("DEBUG:=>putRegisterPageToClient(%s,%s,%s,%s,%s,%s,%s)", 
		      language,macAddress,deviceName,mailAddress,userId,extraId);
  ret=putRegisterPageToClient(language, macAddress, deviceName, mailAddress, userId, extraId);
  if(debug>1) err_msg("DEBUG:(%d)<=putRegisterPageToClient( )",ret);
  return ret;
}

void PutMacCheckListToClient(void){
  if(debug>1) err_msg("DEBUG:=>putMacCheckListToClient( )");
  putMacCheckListToClient();
  if(debug>1) err_msg("DEBUG:<=putMacCheckListToClient( )");
}

void PutMacRegListToClient(char* userId, char* extraId){
  if(debug>1) err_msg("DEBUG:=>putMacRegListToClient( )");
  putMacRegListToClient(userId, extraId);
  if(debug>1) err_msg("DEBUG:<=putMacRegListToClient( )");
}

void PutMessageToClient(char *message){
  if(debug>1) err_msg("DEBUG:=>putMessageToClient(%s)",message);
  putMessageToClient(message);
  if(debug>1) err_msg("DEBUG:<=putMessageToClient( )");
}

int ConvertMacAddr(char* macAddr){
  int ret;
  if(debug>1) err_msg("DEBUG:=>convertMacAddr(%s)", macAddr);
  ret=convertMacAddr(macAddr);
  if(debug>1) err_msg("DEBUG:(%d)<=convertMacAddr(%s)",ret,macAddr);
  return ret;
}

int IsSafeString(char* str, int length){
  int ret;
  if(debug>1) err_msg("DEBUG:=>isSafeString(%s,%d)",str,length);
  ret=isSafeString(str,length);
  if(debug>1) err_msg("DEBUG:(%d)<=isSafeString( )",ret);
  return ret;
}

int HtmlReplace(char* buff,char *beforeStr,char *afterStr){
  int ret;
  if(debug>2) err_msg("DEBUG:=>htmlReplace(%s,%s,%s)",buff,beforeStr,afterStr);
  ret = htmlReplace(buff, beforeStr, afterStr);
  if(debug>2) err_msg("DEBUG:(%d)<=htmlReplace( )",ret);
  return ret;
}

char* StrSplit(char* str,const char* delimStr){
  char* ret;
  if(debug>2) err_msg("DEBUG:=>strSplit(%s,%s)",str,delimStr);
  ret = strSplit(str, delimStr);
  if(debug>2) err_msg("DEBUG:(%s)<=strSplit( )",ret);
  return ret;
}

int GetHttpCookie(char *cookie, char* cookieName){
  int ret;
  if(debug>1) err_msg("DEBUG:=>getHttpCookie(%s)", cookieName);
  ret = getHttpCookie(cookie, cookieName);
  if(debug>1) err_msg("DEBUG:(%d)<=getHttpCookie(%s)",ret, cookie);
  return ret;
}

int IsCorrectCookie(char* cookie, int userType){
  int ret;
  if(debug>1) err_msg("DEBUG:=> isCorrectCookie(%d)", userType);
  ret =  isCorrectCookie(cookie,userType);
  if(debug>1) err_msg("DEBUG:(%d)<= isCorrectCookie(%s)",ret,cookie);
  return ret;
}

int GetUserIdFromEnv(char *userid){
  int ret;
  if(debug>1) err_msg("DEBUG:=>getUserIdFromEnv(%s)",userid);
  ret = getUserIdFromEnv(userid);
  if(debug>1) err_msg("DEBUG:(%d)<=getUserIdFromEnv( )",ret);
  return ret;
}

int GetUserIdFromPostData(char* requestStr, char* userid, char* password){
  int ret;
  if(debug>1) err_msg("DEBUG:=>getUserIdFromPostData(%s)",requestStr);
  ret = getUserIdFromPostData(requestStr,userid,password);
  if(debug>1) err_msg("DEBUG:(%d)<=getUserIdFromPostData(,%s,password)",ret,userid);
  return ret;
}

int PutAuthRequestPageToClient(char *language, char* cgiName){
  int ret;
  if(debug>1) err_msg("DEBUG:=>putAuthRequestPageToClient(%s,%s)",language,cgiName);
  ret=putAuthRequestPageToClient(language,cgiName);
  if(debug>1) err_msg("DEBUG:(%d)<=putAuthRequestPageToClient( )",ret);
  return ret;
}

int AnalyzeUpdateRequestAndExecute(char *content, char* userId, char* extraId){
  int ret;

  if(debug>1) err_msg("DEBUG:=>analyzeUpdateRequestAndExecute(%s,%s,%s)", content,userId,extraId);
  ret=analyzeUpdateRequestAndExecute(content,userId,extraId);
  if(debug>1) err_msg("DEBUG:%d<=analyzeUpdateRequestAndExecute( )",ret);
  return ret;
}

int PutUpdatePageToClient(char *language, char* userId, char* extraId){
  int ret;
  if(debug>1) err_msg("DEBUG:=>putUpdatePageToClient(%s,%s,%s)", 
		      language,userId,extraId);
  ret=putUpdatePageToClient(language,userId,extraId);
  if(debug>1) err_msg("DEBUG:(%d)<=putUpdatePageToClient( )",ret);
  return ret;
}

void PutMacListToClient(char *userId, char* extraId){
  if(debug>1) err_msg("DEBUG:=>putMacListToClient(%s,%s)",userId,extraId);
  putMacListToClient(userId,extraId);
  if(debug>1) err_msg("DEBUG:<=putMacListToClient( )");
}

void PutUsageLogToClient(char *userId, char* extraId, char* language){
  if(debug>1) err_msg("DEBUG:=>putUsageLogToClient(%s,%s,%s)",userId,extraId,language);
  putUsageLogToClient(userId,extraId,language);
  if(debug>1) err_msg("DEBUG:<=putUsageLogToClient( )");
}
