/**************************************************
opengate mac addr auth program
 module for Authentication of User

Copyright (C) 1999 Opengate Project Team
Written by Yoshiaki Watanabe
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Email: watanaby@is.saga-u.ac.jp
**************************************************/

#include "opengatemmng.h"

void onAuthReplyAlarm(int signo);
int authFtp(char *userid, char *passwd);
int AuthFtp(char *userid, char *passwd);
int authPop3(char *userid, char *passwd);
int AuthPop3(char *userid, char *passwd);
int authPam(char *userid, char *passwd);
int AuthPam(char *userid, char *passwd);
int authRadius(char *userid, char *passwd);
int AuthRadius(char *userid, char *passwd);
int authPop3s(char *userid, char *passwd);
int AuthPop3s(char *userid, char *passwd);
int authFtpse(char *userid, char *passwd);
int AuthFtpse(char *userid, char *passwd);
int authFtpsi(char *userid, char *passwd);
int AuthFtpsi(char *userid, char *passwd);
int authLdap(char *userid, char *passwd);
int AuthLdap(char *userid, char *passwd);

/****************************************
get userid from anywhere (cookie/env/postdata)
language indicates the one for web description
****************************************/
int getUserId(char* requestStr, char* userId, char* extraId, char* language, int userType, char* cgiName, char* mailDefault){
  int authResult=DENY;
  char useridfull[USERMAXLN]; /* userid@extraid */
  char cookie[SIDMAXLN];
  char password[USERMAXLN];

  /* default mail address is null */
  mailDefault[0]='\0';

  /* if userid is found in cookie, accept auth */
  switch(userType){
  case NORMALUSER:
    if(GetHttpCookie(cookie, GetConfValue("AuthUserCookie"))){
      if(IsCookieFoundInWorkDb(cookie, userId, extraId, NORMALUSER)){ 
	ConcatUserId(useridfull, userId, extraId);
	GetMailDefaultFromWorkDb(cookie, mailDefault);
	authResult=ACCEPT;
      }
    }
    break;
  case ADMINUSER:
    if(GetHttpCookie(cookie, GetConfValue("AuthAdminCookie"))){
      if(IsCookieFoundInWorkDb(cookie, userId, extraId, ADMINUSER)){ 
	ConcatUserId(useridfull, userId, extraId);
	authResult=ACCEPT;
      }
    }
    break;
  }

  /* if not auth by cookie, check env-var(shibboleth/httpbasic) */ 
  if(authResult==DENY){

    /* search shibboleth / httpbasic auth setting in conf */
    ResetAuthServerPointer();
    while(SelectNextAuthServer()){

      /* if server setting is not matched to the required usertype, skip it */
      if( (strcmp(GetConfValue("AuthServer/UserType"), "admin")==0)
	&& (userType!=ADMINUSER) ) continue;
      if( (strcmp(GetConfValue("AuthServer/UserType"), "admin")!=0)
	&& (userType==ADMINUSER) ) continue;
       
      /* if the server setting is not (shibboleth or httpbasic), skip it */
      if(strcmp(GetConfValue("AuthServer/Protocol"), "shibboleth")!=0
	 && strcmp(GetConfValue("AuthServer/Protocol"), "httpbasic")!=0) continue;
	
      /* if reached to this(=shibboleth/httpbasic), get userid from env var */
      if(GetUserIdFromEnv(useridfull)){

	/* if the user is found in accept user list, accept him, else deny */
	SplitId(useridfull, userId, extraId);
	if(IsUserIdFoundInAcceptUsersList(userId)){
	  authResult=ACCEPT;
	  MakeMailDefault(userId, extraId, mailDefault);
	}
	else{
	  SetMessage(NoInfoInDb); 
	  PutDenyToClient(language);
	  err_msg("DENY: user %s", useridfull);
	  authResult=DENY;
	  return FALSE;
	}
      }
    }
  }

  /* if not auth by cookie and env-var, get user id from request str */
  if(authResult==DENY){
    if(!GetUserIdFromPostData(requestStr, useridfull, password)){
      
      /* if not get user id, send auth request page, and end */
      if(userType==ADMINUSER) SetMessage(RequestAdminAuth);
      PutAuthRequestPageToClient(language, cgiName);
      return FALSE;
    }
  }

  /* if not yet auth, check auth */
  if(authResult==DENY){

    /* split user@extra to user and extra */
    SplitId(useridfull, userId, extraId);
    SetupConfExtra(userId, extraId);

    /* check user by authenticate servers */
    ResetAuthServerPointer();
    while(SelectNextAuthServer()){
      
      /* if check normal user and admin auth server, goto next server */
      /* if check admin user and not admin auth server, goto next server */
      if(userType==NORMALUSER 
	 && strcmp(GetConfValue("AuthServer/UserType"), "admin")==0) continue;
      if(userType==ADMINUSER
	 && strcmp(GetConfValue("AuthServer/UserType"), "admin")!=0) continue;

      /* authenticate the user with auth server. if deny, goto next server */
      if((authResult=AuthenticateUser(userId, password))==DENY) continue;

      /* if user does not exist in the user list, goto next auth server */
      if(!IsUserIdFoundInAcceptUsersList(userId)){
	authResult=DENY;
	continue;
      }

      /* if accepted, set mail default(used for warning mail) end search */
      if(authResult==ACCEPT){
	MakeMailDefault(userId, extraId, mailDefault);
	break;
      }
    }

    /* if fail, put error */
    if(authResult==DENY){
      SetMessage(NoInfoInDb); 
      PutDenyToClient(language);
      err_msg("DENY: user %s", useridfull);
    }
  }
  if(authResult==ACCEPT) return TRUE;
  else return FALSE;
}

/**************************************
 if accept users are listed in conf file,
find the userid in the list
 if no list is indicated, return true
**************************************/
int isUserIdFoundInAcceptUsersList(char* userId){
  char usersList[BUFFMAXLN];
  char userIdPattern[WORDMAXLN];

  /* get accept users list. if not exist, return true(accept all users) */
  strncpy(usersList,GetConfValue("AuthServer/AcceptUsers"),BUFFMAXLN);
  if(isNull(usersList)) return TRUE;

  /* if userid is found in the usersList, return true */
  /* example of usersList is [user1 user2 user3 user4] */

  /* regular expression match to "(^| )userid( |$)" */
  /* meaning is [(head or space) userid-string (space or tail)] */
  strncpy(userIdPattern, "(^| )", WORDMAXLN);
  strncat(userIdPattern, userId, WORDMAXLN);
  strncat(userIdPattern, "( |$)", WORDMAXLN);
  return RegExMatch(usersList, userIdPattern); 
}


/*******************************************************/
/* Authenticate user by accessing to ftp server        */
/*  userid : user to auth                              */
/*  password : password for the user                   */
/*******************************************************/
int authenticateUser(char *userid, char *passwd)
{
  char* proto;
  int authResult=DENY;
  int timeout;

  /* authserver get timeout value */
  timeout=atoi(GetConfValue("AuthServer/Timeout"));

  /* set auth server reply timeout */
  if(timeout>0){
    AddAlarm("AuthReplyAlarm", timeout, TRUE, onAuthReplyAlarm); 
    EnableAlarm();
  }

  /* get Protocol for authentication */
  proto=GetConfValue("AuthServer/Protocol");

  if(strcmp(proto, "ftp")==0){
    /* authenticate by ftp access */
    authResult=AuthFtp(userid, passwd);

  }else if(strcmp(proto, "pop3")==0){
    /* authenticate by pop3 access */
    authResult=AuthPop3(userid, passwd);

  }else if(strcmp(proto, "pam")==0){
    /* authenticate by pam */
    authResult=AuthPam(userid, passwd);

  }else if(strcmp(proto, "radius")==0){
    /* authenticate by radius */
    authResult=AuthRadius(userid, passwd);

  }else if(strcmp(proto, "pop3s")==0){
    /* authenticate by pop3s */
    authResult=AuthPop3s(userid, passwd);

  }else if(strcmp(proto, "ldap")==0){
    /* authenticate by ldap */
    authResult=AuthLdap(userid, passwd);

  }else if(strcmp(proto, "ftpse")==0){
    /* authenticate by ftps-explicit */
    authResult=AuthFtpse(userid, passwd);

  }else if(strcmp(proto, "ftpsi")==0){
    /* authenticate by ftps-implicit */
    authResult=AuthFtpsi(userid, passwd);

  }else if(strcmp(proto, "shibboleth")==0){
    /* deny all request */
    /* if shibboleth auth is accepted, cannot reach at this point */
    authResult=DENY;

  }else if(strcmp(proto, "httpbasic")==0){
    /* deny all request */
    /* if httpbasic auth is accepted, cannot reach at this point */
    authResult=DENY;

  }else if(strcmp(proto, "deny")==0){
    /* deny all request */
    authResult=DENY;

  }else if(strcmp(proto, "accept")==0){
    /* accept all request */
    authResult=ACCEPT;
  }else{
    err_msg("ERR at %s#%d: Unknown protocol:%s",__FILE__,__LINE__,proto);
    authResult=DENY;
  }

  /* stop the auth reply timeout */
  if(timeout>0) RemoveAlarm("AuthReplyAlarm");
  return authResult;
}

/*****************************/
/* split userid and extraId */
/*****************************/
void splitId(char* useridfull, char* userId, char* extraId)
{
  char useridSave[USERMAXLN];

  char* markPnt=NULL;

  strncpy(useridSave, useridfull, USERMAXLN);

  /* separate extraId from userid */
  markPnt=strchr(useridSave, *GetConfValue("UserIdSeparator"));
  if(markPnt==NULL){  
    /* separator mark not found */
    strncpy(extraId,"",USERMAXLN);
    strncpy(userId,useridSave,USERMAXLN);
  }else{
    /* pick up extraId */
    *markPnt='\0';
    strncpy(extraId,markPnt+1,USERMAXLN);
    strncpy(userId,useridSave,USERMAXLN);
  }
} 

/*************************
concatinate userid and extraid
*************************/
char* concatUserId(char* useridfull, char* userId, char* extraId){

  /* set full userid */
  strncpy(useridfull, userId,USERMAXLN);
  if(!isNull(extraId)){
    strncat(useridfull, GetConfValue("UserIdSeparator"), USERMAXLN);
    strncat(useridfull, extraId, USERMAXLN);
  }
  return useridfull;
}

/***********************/
/* Authenticate by FTP */
/***********************/
int authFtp(char *userid, char *passwd)
{
  int		sockfd, n;
  char		recvline[BUFFMAXLN];
  int           authResult;
  char* serverAddr; /* auth server address */
  char* port;      /* auth server port */

  /* get auth server address */
  serverAddr=GetConfValue("AuthServer/Address");

  if(isNull(serverAddr)){
    err_msg("ERR at %s#%d: Missing address for FTP server in config",
	    __FILE__,__LINE__);
    return DENY;
  }

  /* get auth server port */
  port=GetConfValue("AuthServer/Port");

  /* FTP server connect */
  if(isNull(port)){
    sockfd = Tcp_connect(serverAddr, "ftp");
  }else{
    sockfd = Tcp_connect(serverAddr, port);
  }
  if(sockfd<0){
    err_msg("ERR at %s#%d: Ftp server is not normal 0",__FILE__,__LINE__);
    return DENY;
  }

  /* get [220 <host> FTP server ..]*/
  if((n = readln(sockfd, recvline, BUFFMAXLN)) < 0) {
    err_msg("ERR at %s#%d: Ftp server is not normal 1",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }
  if(strstr(recvline,"220")!=recvline){
    err_msg("ERR at %s#%d: Ftp server is not normal 2",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }

  /* put [user <userid>] */
  Writefmt(sockfd, "user %s\r\n", userid);

  /* get [331 Password required ..] */
  if((n = readln(sockfd, recvline, BUFFMAXLN)) < 0) {
    err_msg("ERR at %s#%d: Ftp server is not normal 3",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }

  /* if multi-line greeting [220 ...] exist, skip them. */
  while(strstr(recvline,"220")==recvline){
    if((n = readln(sockfd, recvline, BUFFMAXLN)) < 0) {
      err_msg("ERR at %s#%d: Ftp server is not normal 3",__FILE__,__LINE__);
      Close(sockfd);
      return DENY;
    }
  }

  /* check [331 Password required ..] */
  if(strstr(recvline,"331")!=recvline){
    err_msg("ERR at %s#%d: Ftp server is not normal 4",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }

  /* put [pass <password>] */
  Writefmt(sockfd, "pass %s\r\n", passwd);

  /* get [230 User <userid> logged in] */
  if((n = readln(sockfd, recvline, BUFFMAXLN)) < 0) {
    err_msg("ERR at %s#%d: Ftp server is not normal 5",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }
  if(strstr(recvline,"230")==recvline){
    authResult=ACCEPT;
  }else{
    authResult=DENY;
  }
  
  /* put [quit] */
  Writefmt(sockfd,"quit\r\n");

  Close(sockfd);
  return authResult;
}

/************************/
/* Authenticate by POP3 */
/************************/
int authPop3(char *userid, char *passwd)
{
  int		sockfd, n;
  char		recvline[BUFFMAXLN];
  int           authResult;
  char*  serverAddr;
  char* port;

  /* get auth server address */
  serverAddr=GetConfValue("AuthServer/Address");

  if(isNull(serverAddr)){
    err_msg("ERR at %s#%d: Missing address for POP3 server in config",
	    __FILE__,__LINE__);
    return DENY;
  }

  /* get auth server port */
  port=GetConfValue("AuthServer/Port");

  /* POP3 server connect */
  if(isNull(port)){
    sockfd = Tcp_connect(serverAddr, "pop3");
  }else{
    sockfd = Tcp_connect(serverAddr, port);
  }
  if(sockfd<0){
    err_msg("ERR at %s#%d: Pop3 server is not normal 0",__FILE__,__LINE__);
    return DENY;
  }

  /* get [+OK POP3 <host> <ver> server ready]*/
  if((n = readln(sockfd, recvline, BUFFMAXLN)) < 0) {
    err_msg("ERR at %s#%d: Pop3 server is not normal 1",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }
  if(strstr(recvline,"+OK")!=recvline){
    err_msg("ERR at %s#%d: Pop3 server is not normal 2",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }

  /* put [user <userid>] */
  Writefmt(sockfd, "user %s\r\n", userid);

  /* get [+OK User name accepted, password please] */
  if((n = readln(sockfd, recvline, BUFFMAXLN)) < 0) {
    err_msg("ERR at %s#%d: Pop3 server is not normal 3",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }
  if(strstr(recvline,"+OK")!=recvline){
    err_msg("ERR at %s#%d: Pop3 server is not normal 4",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }

  /* put [pass <password>] */
  Writefmt(sockfd, "pass %s\r\n", passwd);

  /* get [+OK Mailbox open, <count> messages] */
  if((n = readln(sockfd, recvline, BUFFMAXLN)) < 0) {
    err_msg("ERR at %s#%d: Pop3 server is not normal 5",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }
  if(strstr(recvline,"+OK")==recvline){
    authResult=ACCEPT;
  }else{
    authResult=DENY;
  }
  
  /* put [quit] */
  Writefmt(sockfd,"quit\r\n");

  Close(sockfd);
  return authResult;
}

/***************************************/
/* called at auth reply timeout        */
/***************************************/
void onAuthReplyAlarm(int signo)
{
}


/***********************************
check duplication of shibboleth / httpbasic auth server
shibboleth/httpbasic cannot be used for both usertype
***********************************/
int isShibOrBasicAuthDuplicated(void){
  int nShib=0;
  int nBasic=0;

  /* scan config file to count shibboleth/httpbasic auth server setting */
  ResetAuthServerPointer();
  while(SelectNextAuthServer()){
    if(strcmp(GetConfValue("AuthServer/Protocol"), "shibboleth")==0) nShib++;
    if(strcmp(GetConfValue("AuthServer/Protocol"), "httpbasic")==0) nBasic++;
  }
  ResetAuthServerPointer();

  /* if duplicated, put error */
  if(nShib>1 || nBasic>1){
    err_msg("ERR at %s#%d: Shibboleth or HttpBasic auth cannot duplicate",__FILE__,__LINE__);
    return TRUE;
  }else{
    return FALSE;
  }
}

/**************************************
setup default mail address
made from conf value at MailAttribute or MailDomain
**************************************/
int makeMailDefault(char* userId, char* extraId, char* mailDefault){

  char* pMail=NULL;

  /* set default null */
  strncpy(mailDefault, "", BUFFMAXLN);

  /* if set mail attribute, mail from the env variable in attr */
  pMail=GetConfValue("AuthServer/MailAttribute");
  if(!isNull(pMail)){
    pMail=getenvEx(pMail,TRUE,FALSE); /* var list is permitted */
    if(!isNull(pMail)){
      strncpy(mailDefault, pMail, BUFFMAXLN);
      return TRUE;
    }
  }

  /* if set mail domain, mail from userid+@+maildomain */
  pMail=GetConfValue("AuthServer/MailDomain");
  if(!isNull(pMail)){
    strncpy(mailDefault, userId, BUFFMAXLN);
    strncat(mailDefault, "@", BUFFMAXLN);
    strncat(mailDefault, pMail, BUFFMAXLN);
    return TRUE;
  }

  return FALSE;  
}

/**********************************/
/**********************************/
int GetUserId(char* requestStr, char* userId, char* extraId, char* language, int userType, char* cgiName, char* mailDefault){
  int ret;
  if(debug>1) err_msg("DEBUG:=>getUserId(%s,%s,%d)",requestStr,language,userType);
  ret=getUserId(requestStr, userId, extraId, language, userType, cgiName, mailDefault);
  if(debug>1) err_msg("DEBUG:(%d)<=getUserId(%s,%s,%s)",ret,userId,extraId,mailDefault);
  return ret;
}

int AuthFtp(char *userid, char *passwd)
{
  int ret;

  if(debug>1) err_msg("DEBUG:=>authFtp(%s,passwd)",userid);
  ret=authFtp(userid,passwd);
  if(debug>1) err_msg("DEBUG:(%d)<=authFtp( )",ret);

  return ret;
}


int AuthPop3(char *userid, char *passwd)
{
  int ret;

  if(debug>1) err_msg("DEBUG:=>authPop3(%s,passwd)",userid);
  ret=authPop3(userid,passwd);
  if(debug>1) err_msg("DEBUG:(%d)<=authPop3( )",ret);

  return ret;
}

int AuthenticateUser(char *userid, char *password)
{
  int ret;

  if(debug>1) err_msg("DEBUG:=>authenticateUser(%s,passwd)",userid);
  ret=authenticateUser(userid,password);
  if(debug>1) err_msg("DEBUG:(%d)<=authenticateUser( )",ret);

  return ret;
}

void SplitId(char* userid, char* useridshort, char* extraId)
{
  if(debug>1) err_msg("DEBUG:=>splitId(%s,,)",userid);
  splitId(userid,useridshort,extraId);
  if(debug>1) err_msg("DEBUG:<=splitId(,%s,%s)",useridshort,extraId);
}

char* ConcatUserId(char*  useridfull, char* userId, char* extraId)
{
  char* ret;
  if(debug>1) err_msg("DEBUG:=>concatUserId(,%s,%s)",userId,extraId);
  ret=concatUserId(useridfull, userId, extraId);
  if(debug>1) err_msg("DEBUG:<=concatUserId(%s,,)",useridfull);
  return ret;
}

int IsUserIdFoundInAcceptUsersList(char* userId){
  int ret;

  if(debug>1) err_msg("DEBUG:=>isUserIdFoundInAcceptUsersList(%s)",userId);
  ret=isUserIdFoundInAcceptUsersList(userId);
  if(debug>1) err_msg("DEBUG:(%d)<=isUserIdFoundInAcceptUsersList( )",ret);

  return ret;
}

int IsShibOrBasicAuthDuplicated(void){
  int ret;

  if(debug>1) err_msg("DEBUG:=>isShibOrBasicAuthDuplicated( )");
  ret=isShibOrBasicAuthDuplicated();
  if(debug>1) err_msg("DEBUG:(%d)<=isShibOrBasicAuthDuplicated( )",ret);

  return ret;
}

int MakeMailDefault(char* userId, char* extraId, char* mailDefault){
  int ret;

  if(debug>1) err_msg("DEBUG:=>makeMailDefault(%s,%s,)",userId, extraId);
  ret=makeMailDefault(userId, extraId,mailDefault);
  if(debug>1) err_msg("DEBUG:(%d)<=makeMailDefault(,,%s)",ret,mailDefault);

  return ret;
}
