/**************************************************
opengate Mac addr auth program

   module for local work database

Copyright (C) 2011 Opengate Project Team
Written by Yoshiaki Watanabe

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Email: watanaby@is.saga-u.ac.jp
**************************************************/
#include "opengatemd.h"
#include <sqlite3.h>

/********************************************
initialize work db implemented with sqlite
********************************************/
int initWorkDb(void){
  sqlite3 *db;
  char *pErrMsg=NULL;

  /* SQL CREATE COMMAND */
  char *createCmd1="CREATE TABLE IF NOT EXISTS sessionmd "
    "(ipAddress TEXT PRIMARY KEY, "
    "userId TEXT, extraId TEXT, openTime INTEGER, checkTime INTEGER, "
    "macAddress TEXT, ruleNumber INTEGER)";
  char *createCmd2="CREATE TABLE IF NOT EXISTS macinfo "
    "(macAddress TEXT PRIMARY KEY ON CONFLICT REPLACE, "
    "detectTime INTEGER, ttl INTEGER, isNat INTEGER)";

  /* Open sqlite */
  if(sqlite3_open(GetConfValue("SqliteDbMd"),&db)!=SQLITE_OK){
    err_msg("ERR at %s#%d: sqlite3_open",__FILE__,__LINE__);
    sqlite3_close(db);
    terminateProg(0);
  }

  /* create table1 */
  if(sqlite3_exec(db, createCmd1, NULL, NULL, &pErrMsg)!=SQLITE_OK){
    err_msg("ERR at %s#%d: sqlite3_exec: %s",__FILE__,__LINE__,pErrMsg);
    terminateProg(0);
  }

  /* create table2 */
  if(sqlite3_exec(db, createCmd2, NULL, NULL, &pErrMsg)!=SQLITE_OK){
    err_msg("ERR at %s#%d: sqlite3_exec: %s",__FILE__,__LINE__,pErrMsg);
    terminateProg(0);
  }

  /* Sqlite close */
  sqlite3_close(db);

  return TRUE;
}

/************************************************************
 insert session info to db at start     
************************************************************/
int insertSessionToWorkDb(char* ipAddress, char* userId, char* extraId, 
			char* macAddress, int ruleNumber){

  int rc;
  sqlite3 *db;
  char *pErrMsg=NULL;

  /* SQL INSERT COMMAND, where %x is replaced in snprintf */
  char *insertFormat="INSERT INTO sessionmd "
    "(ipAddress, userId, extraId, openTime, checkTime, macAddress, ruleNumber) "
    "values ('%s','%s','%s', %d, %d,'%s', %d)";
  char *insertCmd;
  int resultFlag=TRUE;

  /* Open sqlite */
  if(sqlite3_open(GetConfValue("SqliteDbMd"),&db)!=SQLITE_OK){
    err_msg("ERR at %s#%d: sqlite3_open",__FILE__,__LINE__);
    sqlite3_close(db);
    terminateProg(0);
  }

  /* Prepare insert command */
  insertCmd=sqlite3_mprintf(insertFormat, ipAddress, userId,extraId, 
			    time(NULL), time(NULL), macAddress, ruleNumber);

  /* Execute insert to sqlite */
  if((rc=sqlite3_exec(db, insertCmd, NULL, NULL, &pErrMsg))!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_exec: %s",__FILE__,__LINE__,pErrMsg);
  }

  /*Memory free for sqlite3 string */
  sqlite3_free(insertCmd);

  /* Sqlite close */
  sqlite3_close(db);

  return resultFlag;
}

/*************************************************************
 update checkTime to time now
*************************************************************/
int updateCheckTimeInWorkDb(char* ipAddress){

  sqlite3 *db;
  char *pErrMsg=NULL;

  /* SQL UPDATE COMMAND, where %x is replaced in mprintf */
  char *updateFormat="UPDATE sessionmd "
    "SET checkTime=%d WHERE ipAddress='%s'";
  char *updateCmd;
  int resultFlag=TRUE;

  /* open sqlite */
  if(sqlite3_open(GetConfValue("SqliteDbMd"),&db)!=SQLITE_OK){
    err_msg("ERR at %s#%d: sqlite3_open",__FILE__,__LINE__);
    sqlite3_close(db);
    terminateProg(0);
  }
  
  /* prepare command */
  updateCmd=sqlite3_mprintf(updateFormat, time(NULL), ipAddress);

  /* execute update to sqlite */
  if(sqlite3_exec(db, updateCmd, NULL, NULL, &pErrMsg)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_exec: %s",__FILE__,__LINE__,pErrMsg);
  }

  /*memory free for sqlite3 string */
  sqlite3_free(updateCmd);

  /* sqlite close */
  sqlite3_close(db);

  return resultFlag;
}

/*************************************************************
 delete session info to db at stop     
*************************************************************/
int delSessionFromWorkDb(char* ipAddress){

  sqlite3 *db;
  char *pErrMsg=NULL;

  /* SQL DELETE COMMAND, where %x is replaced in mprintf */
  char *deleteFormat="DELETE FROM sessionmd WHERE ipAddress='%s'";
  char *deleteCmd;
  int resultFlag=TRUE;

  /* open sqlite */
  if(sqlite3_open(GetConfValue("SqliteDbMd"),&db)!=SQLITE_OK){
    err_msg("ERR at %s#%d: sqlite3_open",__FILE__,__LINE__);
    sqlite3_close(db);
    terminateProg(0);
  }

  /* prepare command */
  deleteCmd=sqlite3_mprintf(deleteFormat, ipAddress);

  /* execute delete */
  if(sqlite3_exec(db, deleteCmd, NULL, NULL, &pErrMsg)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_exec: %s",__FILE__,__LINE__,pErrMsg);
  }

  /*memory free for sqlite3 string */
  sqlite3_free(deleteCmd);

  /* sqlite close */
  sqlite3_close(db);
  return TRUE;
}

/************************************************
 get info for active session about ip addr from db  
input = ipAddress, output = others
*************************************************/
int getSessionFromWorkDb(char* ipAddress, char* userId, char* extraId, 
			 int* openTime, int* checkTime, char *macAddress,
			 int* ruleNumber){

  sqlite3 *db;
  sqlite3_stmt *stmt;
 
  /* SQL SELECT COMMAND, where %x is replaced in snprintf */
  char *selectFormat="SELECT userId, extraId, openTime, checkTime, "
    "macAddress, ruleNumber FROM sessionmd WHERE ipAddress='%s'";
  char *selectCmd;
  int resultFlag=TRUE;

  /* open sqlite */
  if(sqlite3_open(GetConfValue("SqliteDbMd"),&db)!=SQLITE_OK){
    err_msg("ERR at %s#%d: sqlite3_open",__FILE__,__LINE__);
    sqlite3_close(db);
    return FALSE;
  }

  /* prepare command string */
  selectCmd=sqlite3_mprintf(selectFormat, ipAddress);
  
  /* compile to internal statement */
  if(sqlite3_prepare(db, selectCmd, BUFFMAXLN, &stmt, NULL)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_prepare",__FILE__,__LINE__);

    /* finalize */
    sqlite3_free(selectCmd);
    sqlite3_finalize(stmt);
    sqlite3_close(db);
    return FALSE;
  }

  /* get first record */
  if(sqlite3_step(stmt)==SQLITE_ROW){
    strncpy(userId, (char*)sqlite3_column_text(stmt, 0), USERMAXLN);
    strncpy(extraId, (char*)sqlite3_column_text(stmt, 1), USERMAXLN);
    *openTime=(int)sqlite3_column_int(stmt, 2);
    *checkTime=(int)sqlite3_column_int(stmt, 3);
    strncpy(macAddress, (char*)sqlite3_column_text(stmt, 4), ADDRMAXLN);
    *ruleNumber=(int)sqlite3_column_int(stmt, 5);
    resultFlag=TRUE;
  }else{
    userId[0]='\0';
    extraId[0]='\0';
    *openTime=0;
    *checkTime=0;
    macAddress[0]='\0';
    *ruleNumber=0;
    resultFlag=FALSE;
  }

  /* finalize */
  sqlite3_free(selectCmd);
  sqlite3_finalize(stmt);
  sqlite3_close(db);
  
  return resultFlag;
}


/************************************************
 close session that exceed time limit  
 1. select the record to close firewall 
 2. delete the record
if delayed=FALSE, close all sessions without delay 
*************************************************/
int delUselessSessionsInWorkDb(int delayed){

  sqlite3 *db;
  char *pErrMsg=NULL;
  int uselessLimitTime;

  /* the session is useless, if it doesn't update after this time */
  uselessLimitTime = time(NULL)-atoi(GetConfValue("UselessTimeout"));
 
  /* if delayed is false, all sessions(before now) are deleted */
  if(!delayed) uselessLimitTime = time(NULL);

  /* SQL SELECT COMMAND, where %x is replaced in snprintf */
  char *selectFormat="SELECT ruleNumber, userId, extraId, ipAddress, "
    "macAddress, openTime FROM sessionmd WHERE checkTime<%d";
  char *deleteFormat="DELETE FROM sessionmd WHERE checkTime<%d";
  char *selectCmd;
  char *deleteCmd;
  int resultFlag=TRUE;

  /* open sqlite */
  if(sqlite3_open(GetConfValue("SqliteDbMd"),&db)!=SQLITE_OK){
    err_msg("ERR at %s#%d: sqlite3_open",__FILE__,__LINE__);
    sqlite3_close(db);
    terminateProg(0);
  }

  /* prepare command string for select */
  selectCmd=sqlite3_mprintf(selectFormat, uselessLimitTime);

  /* exec command, callback function = CloseSession */
  if(sqlite3_exec(db, selectCmd, CloseSession, NULL, &pErrMsg)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite_exec:%s",__FILE__,__LINE__, pErrMsg);
  }

  /* prepare command string for update */
  deleteCmd=sqlite3_mprintf(deleteFormat, uselessLimitTime);

  /* exec command  */
  if(sqlite3_exec(db, deleteCmd, NULL, NULL, &pErrMsg)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite_exec:%s",__FILE__,__LINE__, pErrMsg);
  }
    
  /* finalize */
  sqlite3_free(selectCmd);
  sqlite3_free(deleteCmd);
  sqlite3_close(db);

  return resultFlag;
}

/************************************************
get list of sessions from DB (create hash table)
for opengatemd 
 key=ipAddress, value=0 
*************************************************/
int getSessionTableFromWorkDb(DB* sessionTable){
  DBT hashKey;
  DBT hashVal;
  sqlite3 *db=NULL;
  sqlite3_stmt *stmt=NULL;
  int resultFlag=FALSE;
  char ipAddress[ADDRMAXLN];
  int zero=0;
 
  /* SQL SELECT COMMAND */
  char *selectCmd="SELECT ipAddress FROM sessionmd";

  /* open sqlite */
  if(sqlite3_open(GetConfValue("SqliteDbMd"),&db)!=SQLITE_OK){
    err_msg("ERR at %s#%d: sqlite3_open",__FILE__,__LINE__);
    sqlite3_close(db);
    terminateProg(0);
  }

  /* compile to internal statement */
  if(sqlite3_prepare(db, selectCmd, BUFFMAXLN, &stmt, NULL)!=SQLITE_OK){
    err_msg("ERR at %s#%d: sqlite3_prepare",__FILE__,__LINE__);

    /* finalize */
    sqlite3_finalize(stmt);
    sqlite3_close(db);
    return FALSE;
  }

  /* get result rows */
  while(sqlite3_step(stmt)==SQLITE_ROW){
    resultFlag=TRUE;
    strncpy(ipAddress,(char*)sqlite3_column_text(stmt, 0), ADDRMAXLN);

    /* put to hash table */
    hashVal.data = &zero;
    hashVal.size = sizeof(int);    
    hashKey.data = ipAddress;
    hashKey.size = strlen(ipAddress)+1;
    if(sessionTable->put(sessionTable, &hashKey, &hashVal, 0) == -1) {
      err_msg("ERR at %s#%d: fail to put into hash table",__FILE__,__LINE__);
    }
  }

  /* finalize */
  sqlite3_finalize(stmt);
  sqlite3_close(db);

  return resultFlag;
}

/************************************************
get list of sessions from DB (create hash table)
for opengate (NOT OpengateMd)
 key=ipAddress, value=pid(processID)
*************************************************/
int getSessionTableFromOpengateWorkDb(DB* sessionTable){
  DBT hashKey;
  DBT hashVal;
  sqlite3 *db=NULL;
  sqlite3_stmt *stmt=NULL;
  int resultFlag=FALSE;
  char ipAddress4[ADDRMAXLN];
  char ipAddress6[ADDRMAXLN];
  int pid;
 
  /* SQL SELECT COMMAND */
  char *selectCmd="SELECT clientAddr4, clientAddr6, pid "
    "FROM session where closeTime='-'";

  /* open sqlite for opengate (if cannot open, ignore) */
  if(sqlite3_open(GetConfValue("SqliteDb"),&db)!=SQLITE_OK){
    sqlite3_close(db);
    return FALSE;
  }

  /* compile to internal statement */
  if(sqlite3_prepare(db, selectCmd, BUFFMAXLN, &stmt, NULL)!=SQLITE_OK){
    err_msg("ERR at %s#%d: sqlite3_prepare",__FILE__,__LINE__);

    /* finalize */
    sqlite3_finalize(stmt);
    sqlite3_close(db);
    return FALSE;
  }

  /* get result rows */
  while(sqlite3_step(stmt)==SQLITE_ROW){
    resultFlag=TRUE;
    strncpy(ipAddress4,(char*)sqlite3_column_text(stmt, 0), ADDRMAXLN);
    strncpy(ipAddress6,(char*)sqlite3_column_text(stmt, 1), ADDRMAXLN);
    pid=(int)sqlite3_column_int(stmt, 2);

    /* put to hash table */
    /* if set ipv4 addr, enter it to table */
    if(!isNull(ipAddress4)){
      hashVal.data = &pid;
      hashVal.size = sizeof(int);    
      hashKey.data = ipAddress4;
      hashKey.size = strlen(ipAddress4)+1;
      if(sessionTable->put(sessionTable, &hashKey, &hashVal, 0) == -1) {
	err_msg("ERR at %s#%d: fail to put into hash table",__FILE__,__LINE__);
      }
    }

    /* if set ipv6 addr, enter it to table */
    /* in db, this field may contain ipv4 addr */
    if(!isNull(ipAddress6) && strchr(ipAddress6,':')!=NULL){
      hashVal.data = &pid;
      hashVal.size = sizeof(int);    
      hashKey.data = ipAddress6;
      hashKey.size = strlen(ipAddress6)+1;
      if(sessionTable->put(sessionTable, &hashKey, &hashVal, 0) == -1) {
	err_msg("ERR at %s#%d: fail to put into hash table",__FILE__,__LINE__);
      }
    }
  }

  /* finalize */
  sqlite3_finalize(stmt);
  sqlite3_close(db);

  return resultFlag;
}


/**********************************
 put out detected mac info to db 
for checking nat 
**********************************/
int putMacInfoToWorkDb(char* macAddress, int ttl, int isNat){

  int rc;
  sqlite3 *db;
  char *pErrMsg=NULL;

  /* SQL INSERT COMMAND, where %x is replaced in snprintf */
  char *insertFormat="INSERT INTO macinfo "
    "(macAddress, detectTime, ttl, isNat) "
    "values ('%s', %d, %d, %d)";
  char *insertCmd;
  int resultFlag=TRUE;

  /* Open sqlite */
  if(sqlite3_open(GetConfValue("SqliteDbMd"),&db)!=SQLITE_OK){
    err_msg("ERR at %s#%d: sqlite3_open",__FILE__,__LINE__);
    sqlite3_close(db);
    terminateProg(0);
  }

  /* Prepare insert command */
  insertCmd=sqlite3_mprintf(insertFormat, macAddress, time(NULL), ttl, isNat);

  /* Execute insert to sqlite */
  if((rc=sqlite3_exec(db, insertCmd, NULL, NULL, &pErrMsg))!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_exec: %s",__FILE__,__LINE__,pErrMsg);
  }

  /*Memory free for sqlite3 string */
  sqlite3_free(insertCmd);

  /* Sqlite close */
  sqlite3_close(db);

  return resultFlag;
}

/************************************
is the rule number active in opengate session table
************************************/
int isActiveRuleInOpengateWorkDb(int ruleNumber){

  sqlite3 *db;
  sqlite3_stmt *stmt;
 
  /* SQL SELECT COMMAND, where %x is replaced in snprintf */
  char *selectFormat="SELECT * FROM session "
    " WHERE (ruleNumber6=%d or ruleNumber4=%d) and closeTime='-'";
  char *selectCmd;
  int resultFlag=TRUE;

  /* open sqlite for opengate (if cannot open, ignore) */
  if(sqlite3_open(GetConfValue("SqliteDb"),&db)!=SQLITE_OK){
    sqlite3_close(db);
    return FALSE;
  }

  /* prepare command string */
  selectCmd=sqlite3_mprintf(selectFormat, ruleNumber, ruleNumber);
  
  /* compile to internal statement */
  if(sqlite3_prepare(db, selectCmd, BUFFMAXLN, &stmt, NULL)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_prepare",__FILE__,__LINE__);

    /* finalize */
    sqlite3_free(selectCmd);
    sqlite3_finalize(stmt);
    sqlite3_close(db);
    return FALSE;
  }

  /* get first record */
  if(sqlite3_step(stmt)==SQLITE_ROW) resultFlag=TRUE;
  else resultFlag=FALSE;

  /* finalize */
  sqlite3_free(selectCmd);
  sqlite3_finalize(stmt);
  sqlite3_close(db);
  
  return resultFlag;
}


/*************************************************************
 set close time to session table in opengate db
*************************************************************/
int setCloseTimeToOpengateWorkDb(char* ipAddress){

  sqlite3 *db;
  char *pErrMsg=NULL;

  /* SQL UPDATE COMMAND, where %x is replaced in mprintf */
   char *updateFormat="UPDATE session "
     " SET closeTime=datetime(%d,'unixepoch','localtime') "
     " WHERE closeTime='-' and (clientAddr4='%s' or clientAddr6='%s')";
  char *updateCmd;
  int resultFlag=TRUE;

  /* open sqlite for opengate (if cannot open, ignore) */
  if(sqlite3_open(GetConfValue("SqliteDb"),&db)!=SQLITE_OK){
    sqlite3_close(db);
    terminateProg(0);
  }

  /* prepare command */
  updateCmd=sqlite3_mprintf(updateFormat, time(NULL), ipAddress, ipAddress);

  /* execute update */
  if(sqlite3_exec(db, updateCmd, NULL, NULL, &pErrMsg)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_exec: %s",__FILE__,__LINE__,pErrMsg);
  }

  /*memory free for sqlite3 string */
  sqlite3_free(updateCmd);

  /* sqlite close */
  sqlite3_close(db);
  return TRUE;
}

/************************************************
 get info for active session about ip addr from opengate db  
input = ipAddress, output = others
*************************************************/
int getSessionFromOpengateWorkDb(char* ipAddress, char* userid, 
			 char* openTime, char *macAddress, int* ruleNumber){

  sqlite3 *db;
  sqlite3_stmt *stmt;
 
  /* SQL SELECT COMMAND, where %x is replaced in snprintf */
  char *selectFormat4="SELECT userid, openTime, "
    "macAddr, ruleNumber4 FROM session "
    "WHERE clientAddr4='%s' and closeTime='-'";
  char *selectFormat6="SELECT userid, openTime, "
    "macAddr, ruleNumber6 FROM session "
    " WHERE clientAddr6='%s' and closeTime='-'";
  char *selectCmd;
  int resultFlag=TRUE;

  /* open sqlite for opengate (if cannot open, ignore) */
  if(sqlite3_open(GetConfValue("SqliteDb"),&db)!=SQLITE_OK){
    sqlite3_close(db);
    return FALSE;
  }

  /* prepare command string for IPv4 or IPv6 */
  if(strchr(ipAddress, ':')==NULL){
    selectCmd=sqlite3_mprintf(selectFormat4, ipAddress);
  }else{
    selectCmd=sqlite3_mprintf(selectFormat6, ipAddress);
  }
  
  /* compile to internal statement */
  if(sqlite3_prepare(db, selectCmd, BUFFMAXLN, &stmt, NULL)!=SQLITE_OK){
    resultFlag=FALSE;
    err_msg("ERR at %s#%d: sqlite3_prepare",__FILE__,__LINE__);

    /* finalize */
    sqlite3_free(selectCmd);
    sqlite3_finalize(stmt);
    sqlite3_close(db);
    return FALSE;
  }

  /* get first record */
  if(sqlite3_step(stmt)==SQLITE_ROW){
    strncpy(userid, (char*)sqlite3_column_text(stmt, 0), USERMAXLN);
    strncpy(openTime, (char*)sqlite3_column_text(stmt, 1), USERMAXLN);
    strncpy(macAddress, (char*)sqlite3_column_text(stmt, 2), ADDRMAXLN);
    *ruleNumber=(int)sqlite3_column_int(stmt, 3);
    resultFlag=TRUE;
  }else{
    userid[0]='\0';
    *openTime=0;
    macAddress[0]='\0';
    *ruleNumber=0;
    resultFlag=FALSE;
  }

  /* finalize */
  sqlite3_free(selectCmd);
  sqlite3_finalize(stmt);
  sqlite3_close(db);
  
  return resultFlag;
}

/*********************************************************
 routines for debugging output
*********************************************************/
int InitWorkDb(void){
  int ret;
  if(debug>1) err_msg("DEBUG:=>initWorkDb( )");
  ret = initWorkDb();
  if(debug>1) err_msg("DEBUG:(%d)<=initWorkDb( )",ret);
  return ret;
}
int InsertSessionToWorkDb(char* ipAddress, char* userId, char* extraId, 
			char* macAddress, int ruleNumber){
  int ret;
  if(debug>1) err_msg("DEBUG:=>insertSessionToWorkDb(%s,%s,%s,%s,%d)",
		      ipAddress, userId, extraId, macAddress, ruleNumber);
  ret = insertSessionToWorkDb(ipAddress, userId, extraId, macAddress, ruleNumber);
  if(debug>1) err_msg("DEBUG:(%d)<=insertSessionToWorkDb( )",ret);
  return ret;
}

int UpdateCheckTimeInWorkDb(char* ipAddress){
  int ret;
  if(debug>1) err_msg("DEBUG:=>updateCheckTimeInWorkDb(%s)", ipAddress);
  ret = updateCheckTimeInWorkDb(ipAddress);
  if(debug>1) err_msg("DEBUG:(%d)<=updateCheckTimeInWorkDb( )",ret);
  return ret;
}

int DelSessionFromWorkDb(char* ipAddress){
  int ret;
  if(debug>1) err_msg("DEBUG:=>delSessionFromWorkDb(%s)", ipAddress);
  ret = delSessionFromWorkDb(ipAddress);
  if(debug>1) err_msg("DEBUG:(%d)<=delSessionFromWorkDb( )",ret);
  return ret;
}
int GetSessionFromWorkDb(char* ipAddress, char* userId, char* extraId, 
			 int* openTime, int* checkTime, char *macAddress,
			 int* ruleNumber){
  int ret;
  if(debug>1) err_msg("DEBUG:=>getSessionFromWorkDb(%s)", ipAddress);
  ret = getSessionFromWorkDb(ipAddress, userId, extraId, openTime, 
			       checkTime, macAddress, ruleNumber);
  if(debug>1) err_msg("DEBUG:(%d)<=getSessionFromWorkDb(,%s,%s,%d,%d,%s,%d)",
		      ret,userId,extraId,*openTime,*checkTime,macAddress,
		      *ruleNumber);
  return ret;
}

int DelUselessSessionsInWorkDb(int delayed){
  int ret;
  if(debug>1) err_msg("DEBUG:=>delUselessSessionsInWorkDb(%d)", delayed);
  ret=delUselessSessionsInWorkDb(delayed);
  if(debug>1) err_msg("DEBUG:(%d)<=delUselessSessionsInWorkDb( )",ret);
  return ret;
}

int GetSessionTableFromWorkDb(DB* sessionTable){
  int ret;
  if(debug>1) err_msg("DEBUG:=>getSessionTableFromWorkDb( )");
  ret=getSessionTableFromWorkDb(sessionTable);
  if(debug>1) err_msg("DEBUG:(%d)<=getSessionTableFromWorkDb( )", ret);
  return ret;
}

int GetSessionTableFromOpengateWorkDb(DB* sessionTable){
  int ret;
  if(debug>1) err_msg("DEBUG:=>getSessionTableFromOpengateWorkDb( )");
  ret=getSessionTableFromOpengateWorkDb(sessionTable);
  if(debug>1) err_msg("DEBUG:(%d)<=getSessionTableFromOpengateWorkDb( )", ret);
  return ret;
}

int PutMacInfoToWorkDb(char* macAddress, int ttl, int isNat){
  int ret;
  if(debug>1) err_msg("DEBUG:=>putMacInfoToWorkDb(%s,%d,%d)",macAddress,ttl,isNat);
  ret=putMacInfoToWorkDb(macAddress,ttl,isNat);
  if(debug>1) err_msg("DEBUG:(%d)<=putMacInfoToWorkDb( )", ret);
  return ret;
}

int IsActiveRuleInOpengateWorkDb(int ruleNumber){
  int ret;
  if(debug>1) err_msg("DEBUG:=>isActiveRuleInOpengateWorkDb(%d)",ruleNumber);
  ret=isActiveRuleInOpengateWorkDb(ruleNumber);
  if(debug>1) err_msg("DEBUG:(%d)<=isActiveRuleInOpengateWorkDb( )", ret);
  return ret;
}

int SetCloseTimeToOpengateWorkDb(char* ipAddress){
  int ret;
  if(debug>1) err_msg("DEBUG:=>setCloseTimeToOpengateWorkDb(%s)", ipAddress);
  ret=setCloseTimeToOpengateWorkDb(ipAddress);
  if(debug>1) err_msg("DEBUG:(%d)<=setCloseTimeToOpengateWorkDb( )", ret);
  return ret;
}

int GetSessionFromOpengateWorkDb(char* ipAddress, char* userid, 
			 char* openTime, char *macAddress, int* ruleNumber){

  int ret;
  if(debug>1) err_msg("DEBUG:=>getSessionFromOpengateWorkDb(%s)", ipAddress);
  ret = getSessionFromOpengateWorkDb(ipAddress, userid, openTime, 
			       macAddress, ruleNumber);
  if(debug>1) err_msg("DEBUG:(%d)<=getSessionFromOpengateWorkDb(,%s,%s,%s,%d)",
		      ret,userid,openTime,macAddress,*ruleNumber);
  return ret;
}


