/**************************************************
opengate Mac addr auth program

   module for Controlling sessions 

Copyright (C) 2011 Opengate Project Team
Written by Yoshiaki Watanabe

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Email: watanaby@is.saga-u.ac.jp
**************************************************/
#include "opengatemd.h"


void removeSessionUnmatchedToIpfwRule(DB* ruleTable, DB* sessionTable);
void removeIpfwRuleUnmatchedToSession(DB* ruleTable, DB* sessionTable);

/*******************************
add session for the ip address
add ipfw-rule and table-entry, write db and log 
*******************************/
int addSession(char* ipAddress, char* macAddress, char* userId, char* extraId){
  int ruleNumber;

  /* open firewall */
  ruleNumber=OpenClientGate(ipAddress, FALSE, userId, extraId, macAddress); /* NOT FORCED=avoid overlap */

  /* if error, do nothing */
  if(-2<=ruleNumber && ruleNumber<=2) return FALSE;

  /* if previously opened as -ruleNumber (return -ruleNumber), do follows */
  if(ruleNumber<0){
    ruleNumber=-ruleNumber;

    /* if the rule is for opengate, add new rule for opengatemd, else use it */
    if(IsActiveRuleInOpengateWorkDb(ruleNumber)){
      ruleNumber=OpenClientGate(ipAddress, TRUE, userId, extraId, macAddress); /* FORCED=force overlap */
    }
  }    

  /* write to session db */
  InsertSessionToWorkDb(ipAddress, userId, extraId, macAddress, ruleNumber);

  /* write open log to syslog */
  WriteOpenToSyslog(userId, extraId, ipAddress, macAddress);

  /* write log to management db */
  PutOpenToMngDb(macAddress, ipAddress);
  return TRUE;
}

/*******************************
delete session for the ip address
(del ipfw-rule and table-entry, write db and log) 
*******************************/
void delSession(char* ipAddress){
  char userId[USERMAXLN];
  char extraId[USERMAXLN];
  char macAddress[ADDRMAXLN];
  int openTime;
  int ruleNumber;
  int checkTime;
  int success;

  /* get information from session db */
  success=GetSessionFromWorkDb(ipAddress, userId, extraId, &openTime, 
				 &checkTime, macAddress, &ruleNumber);

  /* close firewall */
  if(success)CloseClientGate(ruleNumber);

  /* del from work db */
  DelSessionFromWorkDb(ipAddress);

  /* write close log to syslog */
  WriteCloseToSyslog(userId, extraId, ipAddress, macAddress, openTime);

  /* write close log to management db */
  PutCloseToMngDb(ipAddress);
}

/*******************************
renew session for the ip address
(renew time in table entry) 
*******************************/
void renewSession(char* ipAddress){

  /* renew the check time value */
  UpdateCheckTimeInWorkDb(ipAddress);
}

/*******************************
search db to find ip that is not used for a long time
del ipfw rule, del table entry, write db/log 
*******************************/
void delUselessSessions(void){

  /* scan db to remove old sessions */
  DelUselessSessionsInWorkDb(TRUE);

  /* scan ipfw rule to close unmatched db row */
  CloseUnmatchSessions();
}

/**********************************
CALLBACK FUNCTION
called back from sqlite3_exec 
in sessiondb.c/delUselessSessionsInDb & delAllSessionsInDb
**********************************/
int closeSession(void* pParam, int argc, char *argv[], char* colName[]){

  int ruleNumber;
  char* userId;
  char* extraId;
  char* ipAddress;
  char* macAddress;
  int openTime;

  if(argc < 5) return 1; /* SQLITE_ERROR */
  
  ruleNumber=atoi(argv[0]);
  userId=argv[1];
  extraId=argv[2];
  ipAddress=argv[3];
  macAddress=argv[4];
  openTime=atoi(argv[5]);

  /* close firewall */
  CloseClientGate(ruleNumber);

  /* write close log to syslog */
  WriteCloseToSyslog(userId, extraId, ipAddress, macAddress, openTime);

  /* write close log to management db */
  PutCloseToMngDb(ipAddress);
  return 0; /* SQLITE_OK */
}

/*******************************
search db to find all ip
del ipfw rule, del table entry, write db/log 
*******************************/
void delAllSessions(void){

  /* scan db to remove all sessions. argument indicates delayed flag */
  DelUselessSessionsInWorkDb(FALSE);

  /* scan ipfw rule to close unmatched db row */
  CloseUnmatchSessions();
}

/**********************************************
is the mac/ip address matched pair found
**********************************************/
int isMatchedSessionFound(char* ipAddress, char* macAddressNow){
  int ret;
  char userId[USERMAXLN];
  char extraId[USERMAXLN];
  int openTime;
  int checkTime;
  char macAddress[ADDRMAXLN];
  int ruleNumber;

  /* get info for the ipAddress */
  ret=GetSessionFromWorkDb(ipAddress, userId, extraId, 
			 &openTime, &checkTime, macAddress,
			     &ruleNumber);
  if(ret==FALSE) return FALSE;

  /* if detected mac and saved mac is matched, return true */
  if(strncmp(macAddress, macAddressNow, ADDRMAXLN)==0)return TRUE;

  /* if exist but unmatched pair, delete the session and return false */
  else{
    DelSession(ipAddress);
    return FALSE;
  }
}

/************************************
debug routine for hash table
************************************/
void dumpTable(DB* table){
  DBT hashKey;
  DBT hashVal;
  int ruleNumber=0;
  char ipAddress[ADDRMAXLN]="?";
  int ret;

  memset(&hashKey, 0, sizeof(DBT));
  memset(&hashVal, 0, sizeof(DBT));
  ret=table->seq(table, &hashKey, &hashVal, R_FIRST);
  while(ret==0){

    ruleNumber=*(int*)(hashVal.data);
    strncpy(ipAddress, (char*)hashKey.data, ADDRMAXLN);
    err_msg("%d:%s", ruleNumber, ipAddress);

    /* get next entry */
    ret=table->seq(table, &hashKey, &hashVal, R_NEXT);
  }
}



/************************************************
 close sessions that lost ipfw rule or database entry
( load rules from ipfw and database, then compare ) 
  sessionTable    sessionTable      ipfw
 for opengatemd   for opengate   ruleTable
   key=IpAddr       IpAddr        IpAddr
   val=0           ProcessID     IpfwRule
      ^              ^               ^
      |--------------|               |
             |----- compare key -----|
entry in ruleTable must be found in one of the sessionTables
if exists in ruleTable and not in sessionTable, remove rule
if exists in sessionTable and not in ruleTable, remove session
caution: opengate saves only one of ipv6 addresses in db.
       : the ruleNumber for opengate is stored with negative value
*************************************************/
void closeUnmatchSessions(void){
  DB* ruleTable;
  DB* sessionTable;

  /* prepare hash table for active sessions */
  if((sessionTable = dbopen(NULL, O_CREAT | O_RDWR, 0644, DB_HASH, NULL)) == NULL) {
    err_msg("ERR at %s#%d: fail to open hash table",__FILE__,__LINE__);
    terminateProg(0);
  }

  /* prepare hash table for rule numbers */
  if((ruleTable = dbopen(NULL, O_CREAT | O_RDWR, 0644, DB_HASH, NULL)) == NULL) {
    err_msg("ERR at %s#%d: fail to open hash table",__FILE__,__LINE__);
    terminateProg(0);
  }

  /* get rule numbers from ipfw */
  GetRuleTableFromIpfw(ruleTable);

  /* get active sessions from work db for opengateMd and opengate */
  GetSessionTableFromWorkDb(sessionTable);
  GetSessionTableFromOpengateWorkDb(sessionTable);

  /****debug print
  err_msg("Rule");
  dumpTable(ruleTable);
  err_msg("Session");
  dumpTable(sessionTable);
  ****/

  /* remove unmatched entry */
  removeSessionUnmatchedToIpfwRule(ruleTable, sessionTable);
  removeIpfwRuleUnmatchedToSession(ruleTable, sessionTable);

  /* erase table area */
  sessionTable->close(sessionTable);
  ruleTable->close(ruleTable);
}

/*******************************
remove session in db unmatched to ipfw active rule
 (db entry exists but no ipfw rule is found)
*******************************/
void removeSessionUnmatchedToIpfwRule(DB* ruleTable, DB* sessionTable){
  DBT hashKey;
  DBT hashVal;
  int retRuleTbl;
  int retSesTbl;
  int pid;
  int ruleNumber;
  char ipAddress[ADDRMAXLN];
  char userId[USERMAXLN];
  char extraId[USERMAXLN];
  char macAddress[ADDRMAXLN];
  int openTime;
  char openTimeStr[WORDMAXLN];
  int checkTime;

  /* scan session table to find entry unmatched to rule table */
  /* get first entry of session table */
  memset(&hashKey, 0, sizeof(DBT));
  memset(&hashVal, 0, sizeof(DBT));
  retSesTbl=sessionTable->seq(sessionTable, &hashKey, &hashVal, R_FIRST);
  while(retSesTbl==0){

    /* if found, save data */
    /* if pid>0, the session is controlled by opengate process having pid */
    /* if pid=0, the session is controlled by opengatemd */
    pid = *(int*)(hashVal.data);
    strncpy(ipAddress, (char*)hashKey.data, ADDRMAXLN);

    /* get entry having same key in rule table */
    hashKey.data = ipAddress;
    hashKey.size = strlen(ipAddress)+1;
    memset(&hashVal, 0, sizeof(DBT));
    retRuleTbl=ruleTable->get(ruleTable, &hashKey, &hashVal, 0);

    /* rule does not exist and the session is controlled by opengatemd */
    if(retRuleTbl!=0 && pid==0){
      
      /* write log and close session */
      if(!GetSessionFromWorkDb(ipAddress, userId, extraId, 
			      &openTime, &checkTime, macAddress, &ruleNumber)){
	err_msg("ERR at %s#%d: fail to get session info",__FILE__,__LINE__);
      }else{
	WriteCloseToSyslog(userId, extraId, ipAddress, macAddress, openTime);
	PutCloseToMngDb(ipAddress);
	DelSessionFromWorkDb(ipAddress);
      }
    }

    /* rule does not exist and the session is controlled by opengate */
    if(retRuleTbl!=0 && pid>0){
	
      /* write log and close session */
      if(!GetSessionFromOpengateWorkDb(ipAddress, userId, 
				      openTimeStr, macAddress, &ruleNumber)){
	err_msg("ERR at %s#%d: fail to get session info",__FILE__,__LINE__);
      }else{
	WriteCloseToSyslog(userId, "", ipAddress, macAddress, time(NULL));
	SetCloseTimeToOpengateWorkDb(ipAddress);
      }
    }
    
    /* rule exists and the session is controlled by opengate */
    if(retRuleTbl==0 && pid>0){

      /* does the controlling process exist */
      if(kill(pid,0)==0 || EPERM==errno){
	; /* process exists. do nothing. */
      }

      /* process does not exist. remove session and rule */
      else{
	CloseClientGate(ruleNumber);
	if(!GetSessionFromOpengateWorkDb(ipAddress, userId, 
					openTimeStr, macAddress, &ruleNumber)){
	  err_msg("ERR at %s#%d: fail to get session info",__FILE__,__LINE__);
	}else{
	  WriteCloseToSyslog(userId, "", ipAddress, macAddress, time(NULL));
	  SetCloseTimeToOpengateWorkDb(ipAddress);
	}
      }
    }

    /* get next rule entry */
    retSesTbl=sessionTable->seq(sessionTable, &hashKey, &hashVal, R_NEXT);
  }
}

/***********************************
remove active ipfw rule unmatched to session in db
 (ipfw rule exists but no db entry is found)
***********************************/
void  removeIpfwRuleUnmatchedToSession(DB* ruleTable, DB* sessionTable){
  DBT hashKey;
  DBT hashVal;
  int retRuleTbl;
  int retSesTbl;
  int ruleNumber;
  char ipAddress[ADDRMAXLN];

  /* scan ipfw rule table to find entry unmatched to session table */
  /* get first entry of ipfw rule table */
  memset(&hashKey, 0, sizeof(DBT));
  memset(&hashVal, 0, sizeof(DBT));
  retRuleTbl=ruleTable->seq(ruleTable, &hashKey, &hashVal, R_FIRST);
  while(retRuleTbl==0){

    /* if found in rule table, save data */
    ruleNumber=*(int*)(hashVal.data);
    strncpy(ipAddress, (char*)hashKey.data, ADDRMAXLN);

    /* get entry having same key in session table */
    hashKey.data = ipAddress;
    hashKey.size = strlen(ipAddress)+1;
    memset(&hashVal, 0, sizeof(DBT));
    retSesTbl=sessionTable->get(sessionTable, &hashKey, &hashVal, 0);

    /* if not found in session table, remove entry in ipfw rule */
    if(retSesTbl!=0){
      
      /* opengate cannot save all ipv6 address. */
      /* if the address is ipv6 */
      /* and if the rule number is found in openate db, */
      /* don't remove it */
      if((strchr(ipAddress, ':')!=NULL) &&
	 IsActiveRuleInOpengateWorkDb(ruleNumber)){
	; /* do nothing */
      }else{
	
	/* write log */
	WriteCloseToSyslog("?", "", ipAddress, "?", time(NULL));
	PutCloseToMngDb(ipAddress);
	
	/* remove ipfw rule */
	CloseClientGate(ruleNumber);
      }
    }
    /* get next rule entry */
    retRuleTbl=ruleTable->seq(ruleTable, &hashKey, &hashVal, R_NEXT);
  }
}


/******************************************
write open message to syslog
******************************************/
void writeOpenToSyslog(char* userId, char* extraId, char* ipAddress, char* macAddress){

  if(extraId[0]=='\0'){
    err_msg("OPEN: user %s from %s at %s", 
	    userId, ipAddress, macAddress);
  }else{
    err_msg("OPEN: user %s%s%s from %s at %s", 
	    userId, GetConfValue("UserIdSeparator"), extraId, 
	    ipAddress, macAddress);
  }
}

/******************************************
write close message to syslog
******************************************/
void writeCloseToSyslog(char* userId, char* extraId, char* ipAddress, char* macAddress, int openTime){

  double time_l;
  int hour, min, sec;

  time_l=difftime((int)time(NULL), openTime);
  hour=time_l/60/60;
  min=(time_l-hour*60*60)/60;
  sec=(time_l-hour*60*60-min*60);

  if(extraId[0]=='\0'){
    err_msg("CLOS: user %s from %s at %s ( %02d:%02d:%02d )", 
	    userId, ipAddress, macAddress, hour,min,sec);
  }else{
    err_msg("CLOS: user %s%s%s from %s at %s ( %02d:%02d:%02d )", 
	    userId, GetConfValue("UserIdSeparator"), extraId, 
	    ipAddress, macAddress, hour,min,sec);
  }
}


/**************************************************
 routines for debugging output
 *************************************************/

int AddSession(char* ipAddress, char* macAddress, char* userId, char* extraId){
  int ret;
  if(debug>1) err_msg("DEBUG:=>addSession(%s,%s,%s,%s)",
		      ipAddress, macAddress, userId, extraId);
  ret = addSession(ipAddress, macAddress, userId, extraId);
  if(debug>1) err_msg("DEBUG:(%d)<=addSession( )",ret);
  return ret;
}
void DelSession(char* ipAddress){
  if(debug>1) err_msg("DEBUG:=>delSession(%s)",ipAddress);
  delSession(ipAddress);
  if(debug>1) err_msg("DEBUG:<=delSession( )");
}
void RenewSession(char* ipAddress){
  if(debug>1) err_msg("DEBUG:=>renewSession(%s)",ipAddress);
  renewSession(ipAddress);
  if(debug>1) err_msg("DEBUG:<=renewSession( )");
}
void DelUselessSessions(void){
  if(debug>1) err_msg("DEBUG:=>delUselessSessions()");
  delUselessSessions();
  if(debug>1) err_msg("DEBUG:<=delUselessSessions( )");
}
void DelAllSessions(void){
  if(debug>1) err_msg("DEBUG:=>delAllSessions()");
  delAllSessions();
  if(debug>1) err_msg("DEBUG:<=delAllSessions( )");
}
int CloseSession(void* pParam, int argc, char *argv[], char* colName[]){
  int ret;
  if(debug>1) err_msg("DEBUG:=>closeSession()");
  ret = closeSession(pParam, argc, argv, colName);
  if(debug>1) err_msg("DEBUG:(%d)<=closeSession( )",ret);
  return ret;
}
int IsMatchedSessionFound(char* ipAddress, char* macAddress){
  int ret;
  if(debug>1) err_msg("DEBUG:=>isMatchedSessionFound(%s,%s)",
		      ipAddress, macAddress);
  ret = isMatchedSessionFound(ipAddress, macAddress);
  if(debug>1) err_msg("DEBUG:(%d)<=isMatchedSessionFound( )",ret);
  return ret;
}

void CloseUnmatchSessions(void){
  if(debug>1) err_msg("DEBUG:=>closeUnmatchSessions( )");
  closeUnmatchSessions();
  if(debug>1) err_msg("DEBUG:<=closeUnmatchSessions( )");
}

void WriteOpenToSyslog(char* userId, char* extraId, char* ipAddress, char* macAddress){
  if(debug>1) err_msg("DEBUG:=>writeOpenToSyslog(%s,%s,%s,%s)", userId, extraId, ipAddress, macAddress);
  writeOpenToSyslog(userId, extraId, ipAddress, macAddress);
  if(debug>1) err_msg("DEBUG:<=writeOpenToSyslog( )");
}

void WriteCloseToSyslog(char* userId, char* extraId, char* ipAddress, char* macAddress, int openTime){
  if(debug>1) err_msg("DEBUG:=>writeCloseToSyslog(%s,%s,%s,%s,%d)", userId, extraId, ipAddress, macAddress, openTime);
  writeCloseToSyslog(userId, extraId, ipAddress, macAddress, openTime);
  if(debug>1) err_msg("DEBUG:<=writeCloseToSyslog( )");
}


/********************testmain********************
void testmain(){

  char* ipAddress="192.168.0.120";
  char* macAddress="00:01:02:03:04:05";
  char* userId="watanaby";
  char* extraId="";

  addSession(ipAddress, macAddress, userId, extraId);
  sleep(10);
  renewSession(ipAddress);
  sleep(10);
  delUselessSessions();
  sleep(10);
  delUselessSessions();
  delSession(ipAddress);
}
************************************************/
