/**************************************************
opengate Mac addr auth program

main cgi module 

Copyright (C) 2011 Opengate Project Team
Written by Yoshiaki Watanabe

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Email: watanaby@is.saga-u.ac.jp
**************************************************/
#include	"opengatemd.h"

void sigHupHandler(int sig);
void sigTermHandler(int sig);

int sigHupArrived=FALSE;

/*********************/
/*  main routine     */
/*********************/
int  main(int argc, char **argv)
{
  char ipAddress[ADDRMAXLN];  /* packet source ip address */
  char macAddress[ADDRMAXLN]; /* packet source mac address */
  char userId[USERMAXLN];     /* user id related to the mac address */
  char extraId[USERMAXLN];    /* optional id for the user */
  int macFound;               /* flag: mac address is resistered in db */
  int sessionFound;           /* flag: session for the address exists */
  int consoleMode=FALSE;      /* flag: start with console mode option */
  int endServiceMode=FALSE;   /* flag: start with end service option */
  int reloadServiceMode=FALSE;/* flag: start with reload option */ 
  int stopServiceMode=FALSE;  /* flag: start with stop service option */ 
  int helpMode=FALSE;         /* flag: start with help mode */
  int ttl;                    /* packet ttl(time to live) or hlim(hop limit) */
  int i;                      /* for loop control */
  int uselessCheckTime=0;     /* the last time for useless check */
  int checkInterval;          /* useless session check interval */
  int isNatOrRouter=FALSE;    /* the packet is sent from nat/router */

  /* analyze arguments */
  for(i=1; i<argc; i++){
    if(*argv[i]=='-' && *(argv[i]+1)=='c') consoleMode=TRUE;
    else if(*argv[i]=='-' && *(argv[i]+1)=='e') endServiceMode=TRUE;
    else if(*argv[i]=='-' && *(argv[i]+1)=='r') reloadServiceMode=TRUE;
    else if(*argv[i]=='-' && *(argv[i]+1)=='s') stopServiceMode=TRUE;
    else helpMode=TRUE;
  }

  /* if unknown args, show help and exit */
  if(helpMode){
    ShowHelp(argv[0]);
    terminateProg(0);
  }

  /* drop root privilege */
  seteuid(getuid());

  /* prepare config file */
  if(OpenConfFile()==-1) terminateProg(0);
 
  /* start log */
  errToSyslog(atoi(GetConfValue("Syslog/Enable")));
  openlog(GetConfValue("MacCheckDaemon"), LOG_PID, atoi(GetConfValue("Syslog/Facility")));

  /* initialize config */
  InitConf();

  /* if reloadServiceMode, send HUP to resident daemon */
  if(reloadServiceMode) ReloadDaemon();

  /* if stopServiceMode, stop resident deamon and this process */
  if(stopServiceMode){
    KillDaemon();
    terminateProg(0);
  }

  /* if endServiceMode, stop deamon and close sessions after setup */
  if(endServiceMode) KillDaemon();

  /* get runmode from command argument and set as daemon */
  if(consoleMode)  errToSyslog(FALSE);    /* console mode (no syslog) */
  else   Daemonize();                     /* daemon mode (fork&exit) */

  /* set lock file to prevent overlapped execution */
  if(!LockDaemonLockFile()) terminateProg(0);

  /* set signal functions */
  signal(SIGHUP, sigHupHandler);
  signal(SIGTERM, sigTermHandler);

  /* initializing */
  InitPcap();
  InitIpCache();
  InitWorkDb();
  if(!InitMngDb()) terminateProg(0);
  InitTtlCheck();

  /* if endService is indicated, close all sessions, and exit */
  if(endServiceMode){
    DelAllSessions();
    terminateProg(0);
  }
  
  /* set check interval and remove residue sessions that are useless */
  checkInterval=atoi(GetConfValue("UselessCheckInterval"));
  uselessCheckTime=time(NULL);
  DelUselessSessions();

  /* enter packet inspection loop */
  while(1){

    /* if sig-hup flag is on, reload this program */
    if(sigHupArrived)execlp(argv[0], argv[0], NULL);

    /* get one packet from pcap */
    if( !GetNextPacketFromPcap(ipAddress, macAddress, &ttl) ){

      /* if no packet arrived, */
      /*  check useless sessions at some interval */
      if( time(NULL) - uselessCheckTime > checkInterval ){
	uselessCheckTime = time(NULL);
	DelUselessSessions();
      }
      /* and return to loop top */
      continue;
    }

    /* if local packet, ignore it and return to loop top */
    if(ttl<=1) continue;

    /* if the ip is checked recently, ignore it and return to loop top */
    if( IsRecentlyCheckedIp(ipAddress) ) continue;

    /* only cache timeout packets proceeds to below */

    /* check nat/router and save info to db */
    isNatOrRouter=IsSentViaNatOrRouter(ipAddress, macAddress, ttl);
    if(isNatOrRouter) PutLogAtNatOrRouter(isNatOrRouter,ipAddress,macAddress,ttl);
    PutMacInfoToWorkDb(macAddress, ttl, isNatOrRouter);

    /* check session table and management db */
    sessionFound = IsMatchedSessionFound(ipAddress, macAddress);
    macFound=QueryMacFromMngDb(macAddress, userId, extraId);

    /* if valid mac and no session, start session */
    if(macFound && !sessionFound){
	AddSession(ipAddress, macAddress, userId, extraId);
    }

    /* if no mac and started session, stop session */
    if(!macFound && sessionFound){
      DelSession(ipAddress);
    }

    /* if valid mac and started session, renew check time */
    if(macFound && sessionFound){
      RenewSession(ipAddress);
    }

    /*  check useless sessions at some interval */
    if( time(NULL) - uselessCheckTime > checkInterval ){
      uselessCheckTime = time(NULL);
      DelUselessSessions();
    }
  }

  /* clear data structures (can't reach here, but coded for debugging) */
  FreeIpCache();
  Freequeue();
  ClosePcap();
  CloseConfFile();
  CloseMngDb();

  terminateProg(0);
  return 0;
}

/********************************
 open and lock proc lock file 
 to prevent overlapped daemon exec
********************************/
int lockDaemonLockFile(void){
  
  static int lockFd;
  char str[WORDMAXLN];
 
  /* open process lock file */
  lockFd = open(GetConfValue("DaemonLockFile"), O_RDWR|O_CREAT, 0644);
  
  /* if cannot open or cannot lock, return false */
  if(lockFd<0){
    err_msg("ERR at %s#%d: cannot open daemon lock file:%s",__FILE__,__LINE__,
	    lockFd);
    return FALSE;
  }
  if(lockf(lockFd, F_TLOCK, 0)<0) return FALSE;

  /* record pid */
  snprintf(str, WORDMAXLN, "%d", getpid());
  write(lockFd, str, strlen(str)+1);

  /* normally locked */
  return TRUE;
}


/******************************
daemonize the process
******************************/
void daemonize(void){

  int pid;
  int i;

  /* detach from parent */
  pid = fork();
  if(pid < 0){
   err_msg("ERR at %s#%d: cannot fork",__FILE__,__LINE__);
    terminateProg(0);
  }

  /* parent process exits */
  if(pid > 0) exit(0);

  /* child process runs from here */
  /* set new prosess group */
  setsid();

  /* close all file descriptors */
  for(i=getdtablesize(); i>=0; i--) close(i);

  /* connect stdin/out/err to null */
  i=open("/dev/null", O_RDWR); dup(i); dup(i);

  /* set newly created file permission */
  /* umask(033);   */

  if(debug>0) err_msg("INFO: Deamon started");
}

/****************************************
show help message
****************************************/
void showHelp(char* procName){
   printf("\n");
   printf("firewall control by mac address\n");
   printf(" see detail in Opengate Homepage\n");
   printf("\n");
   printf(" format: %s [arg] \n", procName);
   printf(" arg : -c = run on console (default is daemon)\n");
   printf("     : -e = close all sessions and end service\n");
   printf("     : -r = reload deamon\n");
   printf("     : -s = stop deamon (not close sessions)\n");
 }

/*********************************
signal handler for SIGHUP
*********************************/
void sigHupHandler(int sig){

  if(debug>0)err_msg("INFO: Deamon receives HUP signal");
  sigHupArrived=TRUE;
}

/*********************************
signal handler for SIGTERM
*********************************/
void sigTermHandler(int sig){

  if(debug>0)err_msg("INFO: Deamon receives TERM signal");
  terminateProg(0);
}

/*********************************
kill daemon process
*********************************/
void killDaemon(void){
  FILE* file=NULL;
  struct stat st;
  int pid=0;
  char* lockFileMd;

  /* get lock file name */
  lockFileMd=GetConfValue("DaemonLockFile");

  /* if lock file is not exists, skip */
  if(stat(lockFileMd, &st)!=0){
    ;
  }

  /* read pid from the file */
  else if((file=fopen(lockFileMd, "r"))==NULL){
    err_msg("ERR at %s#%d: cannot open proc lock file:%s",__FILE__,__LINE__
	    ,lockFileMd);
  }

  else if(fscanf(file, "%d", &pid)==0){
    err_msg("ERR at %s#%d: cannot read proc lock file:%s",__FILE__,__LINE__
	    ,lockFileMd);
  }
  if(file!=NULL) fclose(file);

  /* send kill signal to the pid process */
  if(pid!=0){
    seteuid(0);   /* get root privilege */
    kill(pid, SIGKILL);
    seteuid(getuid());   /* drop root privilege */
  }

  /* remove the lockfile */
  remove(lockFileMd);
}

/*********************************
reload daemon process
*********************************/
void reloadDaemon(void){
  FILE* file=NULL;
  struct stat st;
  int pid=0;
  char* lockFileMd;

  /* get lock file name */
  lockFileMd=GetConfValue("DaemonLockFile");

  /* if lock file is not exists, skip */
  if(stat(lockFileMd, &st)!=0){
    ;
  }

  /* read pid from the file */
  else if((file=fopen(lockFileMd, "r"))==NULL){
    err_msg("ERR at %s#%d: cannot open proc lock file:%s",__FILE__,__LINE__
	    ,lockFileMd);
  }

  else if(fscanf(file, "%d", &pid)==0){
    err_msg("ERR at %s#%d: cannot read proc lock file:%s",__FILE__,__LINE__
	    ,lockFileMd);
  }
  if(file!=NULL)fclose(file);

  /* send hup signal to the pid process */
  if(pid!=0){
    seteuid(0);   /* get root privilege */
    kill(pid, SIGHUP);
    seteuid(getuid());   /* drop root privilege */
  }
}

/************************************
 routines for debugging output
 ***********************************/
int LockDaemonLockFile(void){
  int ret;
  if(debug>1) err_msg("DEBUG:=>lockDaemonLockFile( )");
  ret = lockDaemonLockFile();
  if(debug>1) err_msg("DEBUG:(%d)<=lockDaemonLockFile( )",ret);
  return ret;
}

void Daemonize(void){
  if(debug>1) err_msg("DEBUG:=>daemonize( )");
  daemonize();
  if(debug>1) err_msg("DEBUG:<=daemonize( )");
}

void ShowHelp(char* procName){
  if(debug>1) err_msg("DEBUG:=>showHelp(%s)", procName);
  showHelp(procName);
  if(debug>1) err_msg("DEBUG:<=showhelp( )");
}
void KillDaemon(void){
  if(debug>1) err_msg("DEBUG:=>killDaemon( )");
  killDaemon();
  if(debug>1) err_msg("DEBUG:<=killDaemon( )");
}
void ReloadDaemon(void){
  if(debug>1) err_msg("DEBUG:=>reloadDaemon( )");
  reloadDaemon();
  if(debug>1) err_msg("DEBUG:<=reloadDaemon( )");
}
