/**************************************************
opengate Mac addr auth program

module to get parameters from conf file

Copyright (C) 2011 Opengate Project Team
Written by Yoshiaki Watanabe

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Email: watanaby@is.saga-u.ac.jp
**************************************************/
#include "opengatemd.h"
#include "../ezxml/ezxml.h"

#define CONFIG_VERSION "0.1.0"
#define SEPARATOR "/"

int debug=0;
static ezxml_t xmlRoot=NULL;
static ezxml_t xml=NULL;

char *getConfValue(char *name);
char *convertToFacilityRaw(char *pValue);

/**************************************************/
/* Prepare Conf file to use                       */
/* this is called before syslog setup             */
/**************************************************/
int openConfFile(void)
{
  char *s;
  char *errMsg;

  /* parse file and make tree */
  if((xmlRoot = ezxml_parse_file(CONFIGFILE))==NULL){

    /* as the syslog is not prepared, error is send to console */
    printf("Error: Opengatemd configuration file %s is not found\n",CONFIGFILE);

    return -1;
  }

  /* to check error, convert to xml */
  s=ezxml_toxml(xmlRoot);  free(s);
  
  /* if failed, show error message */
  errMsg=(char *)ezxml_error(xmlRoot);

  if(*errMsg!='\0'){
    /* as the syslog is not prepared, error is send to console */
    printf("Error: Opengatemd configuration file %s is illegal:%s\n", CONFIGFILE,errMsg);
    return -1;
  }

  /* check the config file version */ 
  if(isNull(ezxml_attr(xmlRoot, "ConfigVersion"))||
     (strcmp(CONFIG_VERSION, ezxml_attr(xmlRoot, "ConfigVersion"))!=0)){
    printf("Error: Opengatemd configuration file %s has mismatch version\n",CONFIGFILE);
    printf("%s,%s\n", CONFIG_VERSION, ezxml_attr(xmlRoot, "ConfigVersion"));
    return -1;
  }

  /* check the syslog */
  if(atoi(GetConfValue("Syslog/Enable")) &&
     atoi(GetConfValue("Syslog/Facility"))==0){

    /* as the syslog is not prepared, error is send to console */
    printf("Error: correct SYSLOG setting(local0-local7) is not found in Opengatemd configuration file %s\n", CONFIGFILE);

    return -1;
  }

  return 0;
}

/**************************************************/
/*  initialize the Config                         */
/**************************************************/
void initConf(void)
{
  /* as debug flag is used many times, put it in gloval variable */
  debug=atoi(getConfValue("Debug"));
}

/**************************************************/
/* Finish Conf file usage                         */
/**************************************************/
void closeConfFile(void)
{
  if(xmlRoot!=NULL)ezxml_free(xmlRoot);
}


/**************************************************/
/*  get a value for name from Conf file           */
/*  the name[aa/bb/cc] means the path             */
/*  if ID is set, extraSet value is overrided */
/**************************************************/
char *getConfValue(char *name)
{
  char *pValue;
  char *pStr;
  char buff[BUFFMAXLN];

  /* copy name to work area */
  strncpy(buff,name,BUFFMAXLN);

  /* get first token */
  pStr=strtok(buff, SEPARATOR);

  /* set search start to root of tree */
  xml=xmlRoot;

  /* search the tree node for the name */
  while(pStr!=NULL){
    xml=ezxml_child(xml, pStr);
    pStr=strtok(NULL, SEPARATOR);
  }

  /* get the node value */
  pValue= ezxml_txt(xml);

  /* if not get, write error message */
  if(pValue==NULL){
    err_msg("ERR at %s#%d: cannot get %s from conf file",__FILE__,__LINE__,name);
  }

  /* if syslog facility, the id is converted to raw value */
  if(strcmp(name,"Syslog/Facility")==0){
    pValue=convertToFacilityRaw(pValue);
  }

  /* return found value */
  return pValue;
}

/**************************************************/
/*  get the next value as previous call           */
/*  (next node of the lowest level of tree)       */  
/**************************************************/
char *getNextConfValue(void)
{
  char *pValue;

  /* get next node */
  if(xml==NULL) return "";
  xml = ezxml_next(xml);

  /* get the node value */
  pValue= ezxml_txt(xml);

  /* if not get, write error message */
  if(pValue==NULL) return "";

  /* return found value */
  return pValue;
}

/***********************************************/
/* Convart the syslog facility id to raw value */
/***********************************************/
char *convertToFacilityRaw(char *pValue)
{
  static char facility[WORDMAXLN];
  int rawValue;

  if     (strcmp(pValue, "local0")==0) rawValue=LOG_LOCAL0;
  else if(strcmp(pValue, "local1")==0) rawValue=LOG_LOCAL1;
  else if(strcmp(pValue, "local2")==0) rawValue=LOG_LOCAL2;
  else if(strcmp(pValue, "local3")==0) rawValue=LOG_LOCAL3;
  else if(strcmp(pValue, "local4")==0) rawValue=LOG_LOCAL4;
  else if(strcmp(pValue, "local5")==0) rawValue=LOG_LOCAL5;
  else if(strcmp(pValue, "local6")==0) rawValue=LOG_LOCAL6;
  else if(strcmp(pValue, "local7")==0) rawValue=LOG_LOCAL7;
  else rawValue=0;

  snprintf(facility, WORDMAXLN, "%d", rawValue);

  return facility;
}

/***********************************************
 routines for debugging output
***********************************************/
int OpenConfFile(void){
  int ret;
  if(debug>1) err_msg("DEBUG:=>openConfFile( )");
  ret = openConfFile();
  if(debug>1) err_msg("DEBUG:(%d)<=openConfFile( )",ret);
  return ret;
}

void CloseConfFile(void){
  if(debug>1) err_msg("DEBUG:=>closeConfFile( )");
  closeConfFile();
  if(debug>1) err_msg("DEBUG:<=closeConfFile( )");
}

char *GetConfValue(char *name){
  char *ret;
  if(debug>1) err_msg("DEBUG:=>getConfValue(%s)",name);
  ret=getConfValue(name);
  if(debug>1) err_msg("DEBUG:(%s)<=getConfValue( )",ret);
  return ret;
}

void InitConf(void){
  if(debug>1) err_msg("DEBUG:=>initConf( )");
  initConf();
  if(debug>1) err_msg("DEBUG:<=initConf( )");
}


char *GetNextConfValue(void){
  char *ret;
  if(debug>1) err_msg("DEBUG:=>getnextConfValue( )");
  ret=getNextConfValue();
  if(debug>1) err_msg("DEBUG:(%s)<=getConfValue( )",ret);
  return ret;
}
  
