/**************************************************
opengate Mac addr auth program

 module to control ip addresss cache 

  As ip address check by database is time consuming procedure,
  the recently checked addresses are cached and skiped.
  Implemented with HashTable and Queue.
  HashTable has recentry checked ipAddress and checked time.
  If ip is included in table and time is new, ignore checking.
  Queue has ipAddresses odrered by checked time.
  If an old item is found in table, elder items are removed from table.
  The queue controls the remove sequence.


Copyright (C) 2011 Opengate Project Team
Written by Yoshiaki Watanabe

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Email: watanaby@is.saga-u.ac.jp
**************************************************/
#include "opengatemd.h"

int Initqueue(void);
int Enqueue(char* addr);
int Dequeue(char* addr);
int Listqueue(void);

/* Queue to store IpAddress */
struct queueNode{
  char addr[ADDRMAXLN];
  struct queueNode *next;
};
static struct queueNode* queueTail=NULL;
static struct queueNode* queueHead=NULL;

/* HashTable to store IpAddress and Time */
static DB* hashDb;

/* Cache Timeout(seconds) = DB checking interval */
static int cacheTimeout;


/****************************************
initialize IpAddress Cache
****************************************/
void initIpCache(void) {

  DBT hashKey;
  DBT hashVal;
  int timeNull=0;

  /* prepare hash table */
  if((hashDb = dbopen(NULL, O_CREAT | O_RDWR, 0644, DB_HASH, NULL)) == NULL) {
    err_msg("ERR at %s#%d: fail to open hash table",__FILE__,__LINE__);
    terminateProg(0);
  }

  /* ingoring addresses is registered to hash table */
  hashKey.data = "0.0.0.0";
  hashKey.size = strlen((char *)hashKey.data)+1;
  hashVal.data = &timeNull;
  hashVal.size = sizeof(int);    
  if(hashDb->put(hashDb, &hashKey, &hashVal, 0) == -1) {
    err_msg("ERR at %s#%d: fail to put into hash table",__FILE__,__LINE__);
  }
  hashKey.data = "::";
  hashKey.size = strlen((char *)hashKey.data)+1;
  hashVal.data = &timeNull;
  hashVal.size = sizeof(int);    
  if(hashDb->put(hashDb, &hashKey, &hashVal, 0) == -1) {
    err_msg("ERR at %s#%d: fail to put into hash table",__FILE__,__LINE__);
  }

  /* prepare queue */
  Initqueue();

  /* set timeout parameter */
  cacheTimeout=atoi(GetConfValue("CacheTimeout"));
}

/*********************************************
initialize IpAddress Queue
Queue
 HeadNode - DataNode - DataNode - TailNode
 (dummy)                           (dummy)
  ^queueHead                        ^queueTail
*********************************************/
int initqueue(void){

  char addr[ADDRMAXLN];

  /* if not exist, prepare head and tail */
  if(queueHead==NULL){
    queueHead=(struct queueNode*)malloc(sizeof(struct queueNode));
    if(queueHead==NULL){
      err_msg("ERR at %s#%d: fail to malloc",__FILE__,__LINE__);
      terminateProg(0);
    }
    queueTail=(struct queueNode*)malloc(sizeof(struct queueNode));
    if(queueTail==NULL){
      err_msg("ERR at %s#%d: fail to malloc",__FILE__,__LINE__);
      terminateProg(0);
    }
    queueHead->addr[0]='\0';
    queueTail->addr[0]='\0';
    queueHead->next=queueTail;
    queueTail->next=NULL;
  }
  
  /* if exist, reset all */
  else{
    while(Dequeue(addr))
      ;
  }
  return TRUE;
}

/****************************************
Add data to the tail of Queue
 input=addr
****************************************/
int enqueue(char* addr){
  struct queueNode *newNode;

  /* if not prepared, error */
  if(queueHead==NULL){
    err_msg("ERR at %s#%d: queue not init",__FILE__,__LINE__);
    return FALSE;
  }

  /* add item after the tail and set it as new tail*/
  newNode=(struct queueNode*)malloc(sizeof(struct queueNode));
  if(newNode==NULL){
    err_msg("ERR at %s#%d: fail to malloc",__FILE__,__LINE__);
    terminateProg(0);
  }
  strncpy(queueTail->addr, addr, ADDRMAXLN);
  queueTail->next=newNode;
  queueTail=newNode;
  queueTail->addr[0]='\0';
  queueTail->next=NULL;
  return TRUE;
}

/****************************************
Get and remove address data from the head of Queue
 output=addr
****************************************/
int dequeue(char* addr){

  /* set null string as default */
  addr[0]='\0';

  /* if not prepared, error */
  if(queueHead==NULL){
    err_msg("ERR at %s#%d: queue not init",__FILE__,__LINE__);
    return FALSE;
  }
  else if(queueHead->next==NULL){
    err_msg("ERR at %s#%d: queue not init",__FILE__,__LINE__);
    return FALSE;  
  }

  /* if no data, return false */
  else if(queueHead->next==queueTail){
    return FALSE;
  }

  /* get item from the head */
  else {
    struct queueNode *temp;
    temp=queueHead->next;
    queueHead->next=temp->next;
    strncpy(addr, temp->addr, sizeof(temp->addr));
    free(temp);
  }
  return TRUE;
}

/****************************************
Listing IpAddress Queue (for debugging)
****************************************/
int listqueue(void){

  struct queueNode *temp;

  if(queueHead==NULL) return FALSE;
  temp=queueHead->next;
  while(temp->next!=NULL){
    printf("[%s]\n", temp->addr);
    temp=temp->next;
  }
  return TRUE;
}


/****************************************
Is the IpAddress checked recently or not
 input=ipAddress return TRUE if checked recently
****************************************/
int isRecentlyCheckedIp(char* ipAddress){

  int timeNow;
  char storedAddr[ADDRMAXLN];
  int ret;
  int* pTime;
  DBT hashKey;
  DBT hashVal;

  /* get present time */
  timeNow=time(NULL);

  /* get item matched to the ip from hash table */
  hashKey.data = ipAddress;
  hashKey.size = strlen((char *)hashKey.data)+1;
  memset(&hashVal, 0, sizeof(DBT));
  ret=hashDb->get(hashDb, &hashKey, &hashVal, 0);

  /* get success */
  if(ret==0){

    /* if data is null, return not checked */
    pTime=(int*)(hashVal.data);
    if(pTime==NULL) return FALSE;

    /* if ignoring addresses(eg, 0.0.0.0, ::), return true */
    if(*pTime==0) return TRUE;

    /* if recently checked data, return true */
    if( (timeNow-*pTime) < cacheTimeout ) return TRUE;

    /* if timeover, elder items are removed from queue and hashTable */
    /* get data from queue head and inspect the address */
    /* until present address is found. */
    while(Dequeue(storedAddr)){
      hashKey.data=storedAddr;
      hashKey.size = strlen((char *)hashKey.data)+1;
      hashDb->del(hashDb, &hashKey, 0);
      if(strcmp(ipAddress,storedAddr)==0)break;
    }

    /* insert update item to queue and hashTable */
    Enqueue(ipAddress);
    hashVal.data = &timeNow;
    hashVal.size = sizeof(int);    
    if(hashDb->put(hashDb, &hashKey, &hashVal, 0) == -1) {
      err_msg("ERR at %s#%d: fail to put into hash table",__FILE__,__LINE__);
      terminateProg(0);
    }

    /* and return false */
    return FALSE;
  }

  /* if getting from hash fails, insert address and return false */
  if(ret==1){

    /* insert to hash table */
    hashVal.data = &timeNow;
    hashVal.size = sizeof(int);    
    if(hashDb->put(hashDb, &hashKey, &hashVal, 0) == -1) {
      err_msg("ERR at %s#%d: fail to put into hash table",__FILE__,__LINE__);
      terminateProg(0);
    }

    /* insert to queue */
    Enqueue(ipAddress);
    return FALSE;
  }

  /* else error exit */
  err_msg("ERR at %s#%d: fail to get hash table item",__FILE__,__LINE__);
  return FALSE;
}

/****************************************
memory free for IpAddress Cache
****************************************/
void freeIpCache(void) {

  hashDb->close(hashDb);
}

/****************************************
memory free for IpAddress Queue
****************************************/
void freequeue(void){
  char addr[ADDRMAXLN];
  while(Dequeue(addr));
  free(queueHead);
  free(queueTail);
}

/****************************************************
 routines for debugging putput
 ***************************************************/

void InitIpCache(void) {
  if(debug>1) err_msg("DEBUG:=>initIpCache( )");
  initIpCache();
  if(debug>1) err_msg("DEBUG:<=initIpCache( )");
}
int Initqueue(void){
  int ret;
  if(debug>1) err_msg("DEBUG:=>initqueue( )");
  ret = initqueue();
  if(debug>1) err_msg("DEBUG:(%d)<=initqueue( )",ret);
  return ret;

}
int Enqueue(char* addr){
  int ret;
  if(debug>1) err_msg("DEBUG:=>enqueue(%s)", addr);
  ret = enqueue(addr);
  if(debug>1) err_msg("DEBUG:(%d)<=enqueue( )",ret);
  return ret;

}
int Dequeue(char* addr){
  int ret;
  if(debug>1) err_msg("DEBUG:=>dequeue( )");
  ret = dequeue(addr);
  if(debug>1) err_msg("DEBUG:(%d)<=dequque(%s)",ret, addr);
  return ret;

}
int Listqueue(void){
  int ret;
  if(debug>1) err_msg("DEBUG:=>listQueue( )");
  ret = listqueue();
  if(debug>1) err_msg("DEBUG:(%d)<=listQueue( )",ret);
  return ret;

}

int IsRecentlyCheckedIp(char* ipAddress){
  int ret;
  if(debug>1) err_msg("DEBUG:=>isRecentlyCheckedIp(%s)", ipAddress);
  ret = isRecentlyCheckedIp(ipAddress);
  if(debug>1) err_msg("DEBUG:(%d)<=isRecentlyCheckedIp( )",ret);
  return ret;
}

void FreeIpCache(void) {
  if(debug>1) err_msg("DEBUG:=>freeIpCache()");
  freeIpCache();
  if(debug>1) err_msg("DEBUG:<=freeIpCache()");


}

void Freequeue(void){
  if(debug>1) err_msg("DEBUG:=>freeQueue()");
  freequeue();
  if(debug>1) err_msg("DEBUG:<=freequeue()");

}
/**************** test main *************************
int testmain(){

  char addr[ADDRMAXLN];

  initIpCache();

  while(1){

    printf("Enter addr=");
    scanf("%s", addr);

    if(isRecentlyCheckedIp(addr)){
      printf("new\n");
    }else{
      printf("old\n");
    }
    listqueue();
  }
  return 0;
}
*************************************************/

