/*************************************************
Opengate server
  module for communication with client program (javascript) 

Copyright (C) 1999 Opengate Project Team
Written by Yoshiaki Watanabe 1999-2006
Modified Katsuhiko Eguchi, 2005 

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Email: watanaby@is.saga-u.ac.jp
**************************************************/

#include	"opengatesrv.h"

void GetPeerAddr(int sockfd, char *peerAddr);
void SendTerminateReply(void);
void ReadHttpHeaders(void);
void SendReplyToGetHello(void);
void SendHttpKeepPage(char *userid, char *sessionId, char *language, int port, int cookieAuth, char *redirectedUrl);
int SelectAccept(void);
void AcceptHttpReConnect(int timeout);
int MacAddrCheck(int ipStatus,char *clientAddr4, char *clientAddr6, char *macAddr4, char *macAddr6);
int IsRightKey(char *pNowKey, char *sessionId);
void SendHttpReply(char *reply);
void SendHttpNotFound(void);

void OnUsageTimeLimitAlarm(int signo);
void OnCheckBasicAlarm(int signo);
void OnCheckHttpAlarm(int signo);
void OnReadWaitAlarm(int signo);
void OnAjaxWaitAlarm(int signo);
void CheckBasic(void);
void CheckHttp(void);


extern char ruleNumber4[WORDMAXLN];  /* ipfw rule number in string form  */
extern char ruleNumber6[WORDMAXLN];  /* ip6fw rule number in string form */
extern char language[WORDMAXLN]; /* message language */

int ipType=IPV4;                     /* using IP type */
int listenfd[2]; /* file descriptor for listen port */
int hasSock6=TRUE; /* can get the socket for IPv6 */  
int connfd;   /* file descriptor for connection port */
int connectMode = NOCONNECT; /* the TCP connection mode */
char previousHello[BUFFMAXLN];  /* save previous hello request string */

struct AlarmArg{          /* arguments used in on-alarm functions */
  struct clientAddr *pClientAddr;
  char *clientAddr4;
  char *macAddr4;
  char *clientAddr6;
  char *macAddr6;
  char *userid;
  char *userProperty;
  int ipStatus;                    /* ipv4 ipv6 or dual */
  int checkInterval;
  int noPacketInterval;
} alarmArg;

int helloWait=FALSE;  /* hello reply waiting mode */
int readHelloTime=0;  /* the time of reading hello */
int sendHelloTime=0;  /* the time of sending hello */
int noReplyCount=0; /* count up the no reply to hello message */

int checkBasicAlarmRinged=FALSE;
int checkHttpAlarmRinged=FALSE;


/***************************************/
/* get temp listen port of this server */
/***************************************/
int getListenPort(void)
{
  struct sockaddr_in servaddr4;
  struct sockaddr_in6 servaddr6;
  extern const struct in6_addr in6addr_any;

  int portNo;
  int portmin;
  int portmax;
  
  bzero(&servaddr4, sizeof(servaddr4));
  bzero(&servaddr6, sizeof(servaddr6));

  servaddr4.sin_family=AF_INET;
  servaddr4.sin_addr.s_addr=htonl(INADDR_ANY);
  
  servaddr6.sin6_family=AF_INET6;
  servaddr6.sin6_addr=in6addr_any;

  /* get port range from config file */
  portmin=atoi(GetConfValue("ListenPort/Min"));
  portmax=atoi(GetConfValue("ListenPort/Max"));
  
  /* search unused port between PORTMIN and PORTMAX */
  for(portNo=portmin; portNo<=portmax; portNo++){
    servaddr4.sin_port=htons(portNo);
    servaddr6.sin6_port=htons(portNo);

    listenfd[0]=Socket(AF_INET, SOCK_STREAM, 0);
    listenfd[1]=Socket(AF_INET6, SOCK_STREAM, 0);

    if(listenfd[0]<0) return -1; /* if error, return */
    if(listenfd[1]<0) hasSock6=FALSE; /* IPv6 disabled */

    if(hasSock6){
      /* case of socket IPv6 is enabled */
      if(listenfd[0]>=FD_SETSIZE && listenfd[1]>=FD_SETSIZE) return -1;
      
      if((bind(listenfd[0], (SA *)&servaddr4, sizeof(servaddr4))==0) &&
	 (bind(listenfd[1], (SA *)&servaddr6, sizeof(servaddr6))==0) ){
	break;
      }
       
      Close(listenfd[0]);
      Close(listenfd[1]);
    }
    else{
      /* case of socket IPv6 is disabled */
      if(listenfd[0]>=FD_SETSIZE) return -1;
      if(bind(listenfd[0], (SA *)&servaddr4, sizeof(servaddr4))==0)break;
      Close(listenfd[0]);
    }
  }


  if(portNo>portmax) return -1;  /* cannot get unused port */
  
  if(Listen(listenfd[0], LISTENQ)<0) return -1; /* if error, return */
  if(hasSock6){
    if(Listen(listenfd[1], LISTENQ)<0) return -1;
  }
  return portNo;
}

/************************************/
/* wait for connection of client side program */
/************************************/
int waitClientConnect(char *userid, char *userProperty, char *sessionId, char *clientAddr4, char *clientAddr6, int duration, char *macAddr4, char *macAddr6, int ipStatus, struct clientAddr *pClientAddr, char *language, int port, int pid, int cookieAuth, char *redirectedUrl)
{
  char buff[BUFFMAXLN];             /* read in buffer */
  char connectAddr[ADDRMAXLN];      /* connected client address */
  char httpStr[BUFFMAXLN];          /* HTTP GET string at terminate */
  int timeSendHttpkeep=0;
  int timeStart;

  timeStart=time(NULL);

  /* set alarm function arguments */
  alarmArg.pClientAddr=pClientAddr;
  alarmArg.clientAddr4=clientAddr4;
  alarmArg.clientAddr6=clientAddr6;
  alarmArg.macAddr4=macAddr4;
  alarmArg.macAddr6=macAddr6;
  alarmArg.userid=userid;
  alarmArg.userProperty=userProperty;
  alarmArg.ipStatus=ipStatus;
  alarmArg.checkInterval=atoi(GetConfValue("ActiveCheckInterval"));
  alarmArg.noPacketInterval=atoi(GetConfValue("NoPacketInterval"));  

  /* set no conection initially */
  connectMode=NOCONNECT;

  /* set the alarm for usage time limit */
  AddAlarm("UsageTimeLimitAlarm",duration,FALSE,OnUsageTimeLimitAlarm);

  /* set the alarm for periodic keep alive check */
  AddAlarm("CheckBasicAlarm", alarmArg.checkInterval, FALSE, OnCheckBasicAlarm);

  /* loop until accepting correct user */
  while(connectMode == NOCONNECT){

    /* start alarms */
    EnableAlarm();

    /* connection wait */
    connfd = SelectAccept();

    /* stop alarms*/
    DisableAlarm();

    /* if abnormal accept check interupt */
    if(connfd<0){
      
      /* periodic check */
      if(checkBasicAlarmRinged) CheckBasic();

      Close(connfd);
      continue;
    }

    /* is it from the correct client addr(the check is skipped for IPv6) */
    if(ipType==IPV4 && ipStatus!=IPV6ONLY){
      GetPeerAddr(connfd, connectAddr);
      if(isNull(connectAddr)||strcmp(connectAddr, clientAddr4)!=0){
	Close(connfd);
	continue;
      }
    }
 
    /* set read wait alarm */
    AddAlarm("ReadWaitAlarm",atoi(GetConfValue("CommWaitTimeout")),
	     TRUE, OnReadWaitAlarm); 
    EnableAlarm();

    /* get string from connection */
    if(readln(connfd, buff, BUFFMAXLN) <0){
      /* if abnormal, wait next request */
      Close(connfd);
      continue;
    }
    RemoveAlarm("ReadWaitAlarm");
    
    /* is it the httpkeep page download request */
    /* the request is [GET /httpkeep.html ....] */
    snprintf(httpStr, BUFFMAXLN, "GET /httpkeep-%s-%s", userid,sessionId);
    if(strstr(buff, httpStr)==buff){
      
      /* read out the remained headers and send the page */
      ReadHttpHeaders();
      SendHttpKeepPage(userid, sessionId, language, port, cookieAuth, 
		       redirectedUrl);
      timeSendHttpkeep=time(NULL);

      /* return to wait next request (that night be GET /hello) */
      Close(connfd);
      continue;
    }

    /* is it hello request from javascript in httpkeep.html */
    if(strstr(buff, "GET /hello-")==buff){
      
      /* read out the remained headers */
      ReadHttpHeaders();

      /* if the request is premature or too late, ignore */
      if((timeSendHttpkeep == 0)||
	 (time(NULL)-timeSendHttpkeep)> atoi(GetConfValue("CommWaitTimeout"))){
	Close(connfd);
	continue;
      }

      /* if received key is incorrect, ignore */
      /* [GET /hello-key1-key2 ..] */
      if( IsRightKey(buff+strlen("GET /hello-"), sessionId)==FALSE){
	Close(connfd);
	continue;
      }
      
      /* enter to the Http watch mode on the connection keeped(not close) */
      /* reply to the hello request is done in the watch routine */
      connectMode=HTTPCONNECT;
      break;
    }

    /* is it the terminate request */
    /* the request is [GET /terminate-<pid> ..] */
    snprintf(httpStr, BUFFMAXLN, "GET /terminate-%d", pid);
    if(strstr(buff, httpStr)==buff){
      /* terminate request found */
      SendTerminateReply();
      connectMode=ENDCONNECT;
      Close(connfd);
      break;
    }

    /* other request is ignored */
    if(strstr(buff, "GET /")==buff){
      ReadHttpHeaders();
      SendHttpNotFound();
    }
    Close(connfd);
    continue;
  }

  /* stop all alarms */
  RemoveAlarm(NULL);
    
  return connectMode;
}

/****************************/
/* wait for TCP connection  */
/****************************/
int selectAccept(void)
{
  int connfd1 = -1;
  int connfd2 = -1;
  int smax;                         /* select max descliptor */
  fd_set rfd0;                      /* fd_set for select */
  int n;                            /* counter */
  struct sockaddr_storage cliaddr;  /* client IP adddress */
  socklen_t len = sizeof(cliaddr);

  /* select socket */
  FD_ZERO(&rfd0);
  FD_SET(listenfd[0], &rfd0);
  if(hasSock6) FD_SET(listenfd[1], &rfd0);

  if(hasSock6){
    if(listenfd[0]>listenfd[1]) smax=listenfd[0]+1;
    else smax=listenfd[1]+1;
  }else{
    smax=listenfd[0]+1;
  }

  /* wait connection */
  if((n = select(smax, &rfd0, NULL, NULL, NULL)) > 0){

    /* connect by ipv4 */
    if(FD_ISSET(listenfd[0], &rfd0)){
      if((connfd1=accept(listenfd[0],(struct sockaddr *)&cliaddr, &len))>=0){
	ipType=IPV4;
      }
    }

    /* connect by ipv6 */
    if(hasSock6){
      if(FD_ISSET(listenfd[1], &rfd0)){
	if((connfd2=accept(listenfd[1],(struct sockaddr *)&cliaddr, &len))>=0){
	  ipType=IPV6;
	}
      }
    }
  }
  
  if(connfd1>=0 && connfd2>=0){
    Close(connfd2);
    return connfd1;
  }
  else if(connfd1>=0 && connfd2<0) return connfd1;
  else if(connfd1<0 && connfd2>=0) return connfd2;

  return -1;
}

/***************************************/
/* called at usage time limit          */
/***************************************/
void onUsageTimeLimitAlarm(int signo)
{
  connectMode=ENDCONNECT;
  err_msg("ERR at %s#%d: duration timeout",__FILE__,__LINE__);
}

/***************************************/
/* called at read wait time limit       */
/***************************************/
void onReadWaitAlarm(int signo)
{
  connectMode=ENDCONNECT;
}

/***************************************/
/* called at ajax request wait timeout */
/***************************************/
void onAjaxWaitAlarm(int signo)
{
  connectMode=NOCONNECT;
}

/***************************************/
/* called at periodic alive basic check */
/***************************************/
void onCheckBasicAlarm(int signo)
{
  checkBasicAlarmRinged=TRUE;
}

/***************************************/
/* called at periodic alive basic check */
/***************************************/
void checkBasic(void)
{
  static int packetCountPrev=0;  /* packet count at previous check */
  int packetCountNow=0;   /* packet count at now */
  static int noPacketIntervalCount=0; /* no packet count in check loop */
  static int firstCheck=TRUE;
  int duplicateRule4,duplicateRule6;
  int otherPid;

  /* search new IPv6 addresses */
  ScanNdpEntry(alarmArg.pClientAddr, alarmArg.userid,
	     alarmArg.macAddr6, alarmArg.userProperty);

  /* mac address check */
  if(MacAddrCheck(alarmArg.ipStatus, 
		  alarmArg.clientAddr4, alarmArg.clientAddr6, 
		  alarmArg.macAddr4, alarmArg.macAddr6)==FALSE){
    connectMode=ENDCONNECT;
    return;
  }  

  /* packet flow check */
  packetCountNow=GetPacketCount(alarmArg.pClientAddr);
  if(packetCountNow==packetCountPrev){  /* no packet between checks */
    noPacketIntervalCount++;
  }else{
    noPacketIntervalCount=0;
    packetCountPrev=packetCountNow;
  }
  
  if(noPacketIntervalCount*alarmArg.checkInterval 
     >= alarmArg.noPacketInterval){
    err_msg("ERR at %s#%d: no packet passed for the client",
	    __FILE__,__LINE__);
    connectMode=ENDCONNECT;
    return;
  }

  /* duplicated session check */
  /* this is added for ios behavior */
  /* (duplication of ios web-auth and user web-auth) */
  /* if dup, the process in basic mode is ended */
  if(firstCheck){
    firstCheck=FALSE;
    if( FindDuplicateInDbAndClose(alarmArg.clientAddr4, 
				  &duplicateRule4, &duplicateRule6, &otherPid) ){
      
      /* the process exists (ret==0 means the existence) */
      if(kill(otherPid, 0)==0){ 

	/* remove my rules and return as duplicated */
	RemoveDuplicateRule(duplicateRule4, duplicateRule6);
	err_msg("ERR at %s#%d: duplicated ipfw rules for %s are removed",
		__FILE__,__LINE__, alarmArg.clientAddr4);
	connectMode=DUPLICATED;
      }

      /* the process does not exist */
      else{ 
	FixProcessEndInDb(otherPid, "NONE");
      }
    }
  }

  /* set the alarm for next periodic keep alive check */
  checkBasicAlarmRinged=FALSE;
  AddAlarm("CheckBasicAlarm", alarmArg.checkInterval, FALSE, OnCheckBasicAlarm);  /* EnableAlarm is called automatically in alarm function */
}

/***************************************/
/* check mac address change            */
/***************************************/
int macAddrCheck(int ipStatus, char *clientAddr4, char *clientAddr6, char *macAddr4, char *macAddr6) 
{ 
  char macAddrNow[ADDRMAXLN];       /* MAC address at now */

  if(ipStatus!=IPV6ONLY){
    /* check mac address from arp */
    GetMacAddrFromArp(clientAddr4, macAddrNow);
    if(*macAddrNow!='?' && strcmp(macAddr4, macAddrNow)!=0){
      err_msg("ERR at %s#%d: mac address is changed",__FILE__,__LINE__);
      connectMode=ENDCONNECT;
      return FALSE;
    }
  }else{
    /* check mac address from ndp */
    GetMacAddrFromNdp(clientAddr6, macAddrNow);
    if(*macAddrNow!='?' && strcmp(macAddr6, macAddrNow)!=0){
      err_msg("ERR at %s#%d: mac address is changed",__FILE__,__LINE__);
      connectMode=ENDCONNECT;
      return FALSE;
    }
  }
  return TRUE;
}

/***************************************/
/* get addr of connected remote site   */
/***************************************/
void getPeerAddr(int sockfd, char *peerAddr)
{
  struct sockaddr *cliaddr;
  socklen_t len;
  char *pAddr;

  *peerAddr='\0'; /* set null string */

  if((cliaddr=Malloc(ADDRMAXLN))==NULL) return; /* if error, return */

  len=ADDRMAXLN;
  if(Getpeername(sockfd, cliaddr, &len)<0) return; /* if error, return */
    
  pAddr=Sock_ntop_host(cliaddr, len);
  if(pAddr!=NULL) strlcpy(peerAddr, pAddr, ADDRMAXLN);
  
  free(cliaddr);
  
  return;
}


/***************************************************/
/* send reply to terminate access via connfd(not stdout) */
/***************************************************/
void sendTerminateReply(void)
{
  /* send HTTP headers */
  Writefmt(connfd,"HTTP/1.1 200 OK\r\n");
  Writefmt(connfd,"Connection: Close\r\n");
  Writefmt(connfd,"Content-Type: text/html\r\n");
  Writefmt(connfd,"Content-Length: 92\r\n");
  Writefmt(connfd,"\r\n");
  
  Writefmt(connfd,"<META HTTP-EQUIV=Pragma CONTENT=no-cache>");
  Writefmt(connfd,"<HTML><BODY> Network is closed. </BODY></HTML> \r\n\r\n");
  Close(connfd);
}

/***************************************************/
/* send quit to client and close connection        */
/***************************************************/
void sendQuitClient(void)
{
  Writefmt(connfd,"quit\r\n");
  Close(connfd);
}


/***************************************************/
/* wait for close connection of HTTP connection    */
/***************************************************/
void waitHttpClose(struct clientAddr *pClientAddr, char *userid, char *userProperty, char *macAddr4, char *macAddr6, int ipStatus, char *sessionId, int port)
{
  char buff[BUFFMAXLN];
  int ret;

  /* set alarm function arguments */
  alarmArg.pClientAddr=pClientAddr;
  alarmArg.macAddr4=macAddr4;
  alarmArg.macAddr6=macAddr6;
  alarmArg.userid=userid;
  alarmArg.userProperty=userProperty;
  alarmArg.ipStatus=ipStatus;
  alarmArg.checkInterval=atoi(GetConfValue("ActiveCheckInterval"));

  /* save time to read hello */
  readHelloTime = time(NULL);

  /* send first reply to hello */
  SendReplyToGetHello();
  sendHelloTime = time(NULL);

  /* set the hello wait mode ON */
  helloWait=TRUE;
  
  /* TCP read/write loop */
  /* this loop implement following logic */
  /*  repeat until receiving 'GET /terminate' or EOF from client */
  /*      wait request from client.                */
  /*      if wait timeout,  then quit              */
  /*      if 'GET /hello' request, then send reply */
  /*      if 'GET /terminate' request,  then quit  */
  /*      if connection closed, then quit          */
  
  /* set the alarm for periodic client check */
  AddAlarm("checkHttpAlarm", alarmArg.checkInterval, FALSE, OnCheckHttpAlarm);
  EnableAlarm();
  
    /* loop unitl end connection */  
  while(connectMode!=ENDCONNECT){

    /* set default to unknown */
    connectMode=NOCONNECT;

    /* read in wait for client request */
    ret=readln(connfd, buff, BUFFMAXLN);

    /* at some request, stop alarm between readin check*/
    DisableAlarm();

    if(ret>=0){
      /* normal read */

      /* read [GET /hello] */
      if(strstr(buff,"GET /hello") == buff){
	
	ReadHttpHeaders();

	/* if retry request (having same key as previous) */
	if(strncmp(buff, previousHello, BUFFMAXLN)==0) {

	  /*SendReplyToGetHello();*/
	  /* some mulfunction on communication. reset connection */
	  connectMode=NOCONNECT;
	  Close(connfd);
	  AcceptHttpReConnect(atoi(GetConfValue("CommWaitTimeout")));
	  continue;
	}

	/* save hello request string including keys */	  
	strlcpy(previousHello, buff, BUFFMAXLN);

	/* if received key is correct, save time */
	if(IsRightKey(buff+strlen("GET /hello-"), sessionId)){
	
	  /* recognize HTTP hello request */
	  connectMode=HTTPCONNECT;
	
	  /* save time to read hello */
	  readHelloTime = time(NULL);

	  /* set the hello wait mode OFF */
	  helloWait=FALSE;
	}
      
	/* if key is incorrect, terminate */
	else{
	  connectMode=ENDCONNECT;
	}
      }

      /* read other string */
      else{
	
	/* read request and reply not found */
	if(strstr(buff,"GET /") == buff){
	  ReadHttpHeaders();
	  SendHttpNotFound();
	}
	err_msg("ERR at %s#%d: Unknown request",__FILE__,__LINE__);
	connectMode=NOCONNECT;
	Close(connfd);
	AcceptHttpReConnect(atoi(GetConfValue("CommWaitTimeout")));
      }
    }

    else{
      /*abnormal read */
      /* some alarm is ringed or connecion is closed */

      /* periodic check */
      if(checkHttpAlarmRinged) CheckHttp();

      /* if no connection, retry connect */
      if(connectMode==NOCONNECT){

	/* wait short time to accept reconnection */
	/* (to permit occasional disconnection at exchanging hello) */
	Close(connfd);
	if(helloWait==TRUE){
	  AcceptHttpReConnect(atoi(GetConfValue("ReconnectTimeout")));
	}else{
	  AcceptHttpReConnect(atoi(GetConfValue("CommWaitTimeout")));
	}
      }
    }

    /* restart alarm */
    EnableAlarm();
  }

  /* reset alarm */
   RemoveAlarm(NULL);

  Close(connfd);
  
  return;
}

/*************************/
/* accept http reconnect */
/*************************/
void acceptHttpReConnect(int timeout){

  int startTime; 

  startTime=time(NULL);

  /* timeout value should be positive */
  if(timeout<=0){
    connectMode=ENDCONNECT;
    return;
  }

  /* wait a short time */
  AddAlarm("ReadWaitAlarm", timeout, TRUE, OnReadWaitAlarm); 
  EnableAlarm();

  /* connection wait */
  connfd = SelectAccept();

  /* receive request */
  RemoveAlarm("ReadWaitAlarm");
  if(connfd<0){  /* timeout or abnormal */
    connectMode=ENDCONNECT;
  }else{         /* reconnected */
    connectMode=HTTPCONNECT;
    RemoveAlarm("CheckHttpAlarm");
    AddAlarm("CheckHttpAlarm", alarmArg.checkInterval, FALSE, OnCheckHttpAlarm);
  }
}

/*****************************************************************/
/* Is the sent client key correct                                */
/*  request string is as follows                                 */
/*  GET /hello-11111111111111111111-2222222222222222222 HTTP...  */
/*              MD5digest(32chars)   MD5digest(32chars)          */
/*                nowKey               nextKey                   */
/*                                                               */
/*         client                            server              */
/*       save sid     <-------------------     sid               */
/* md5(sid)+md5(md5(rand1)+sid)  ---------> check nowkey         */
/*                                          save nextKey         */
/* md5(rand1)+md5(md5(rand2)+sid)---------> check nowkey         */
/*                                          save nextKey         */
/* md5(rand2)+md5(md5(rand3)+sid)---------> check nowkey         */
/*                                          save nextKey         */
/*****************************************************************/
int isRightKey(char *arg, char *sessionId)
{
  static char savedKey[33]=""; /* saved MD5 string */
  char tempbuff[BUFFMAXLN];    /* work area */
  char md5work[33]      ;      /* md5 work */
  char *pNowKey;
  char *pNextKey;

  /* initial value of savedKey is md5(md5(sessionId)+sessionId) */
  if(isNull(savedKey)){
    md5hex(tempbuff, 33, sessionId);
    strlcat(tempbuff, sessionId, BUFFMAXLN);
    md5hex(savedKey, 33, tempbuff);
  }

  /* split NowKey and NextKey in argument */
  /* 32 is the length of MD5 result */
  pNowKey=arg;
  *(pNowKey+32)='\0';
  pNextKey=pNowKey+33;
  *(pNextKey+32)='\0';

  /* make string [nowKey+sessionId] */
  strlcpy(tempbuff, pNowKey, BUFFMAXLN);
  strlcat(tempbuff, sessionId, BUFFMAXLN);

  /* compare savedKey and md5(nowKey+sessionId) */
  if(strcmp(savedKey, md5hex(md5work, 33, tempbuff))==0){

    /* save nextKey for next check */
    strlcpy(savedKey, pNextKey, 33);
    return TRUE;
  }
  else{
    return FALSE;
  }
}

/***************************************/
/* called at periodic http alive check */
/***************************************/
void onCheckHttpAlarm(int signo)
{
  checkHttpAlarmRinged=TRUE;
}

/***************************************/
/* called at periodic http alive check */
/***************************************/
void checkHttp(void)
{
  /* at this timing, hello request might be received */
  /* if in hello wait mode, it is abnormal */
  if(helloWait==TRUE){
    connectMode=NOCONNECT;
    return;
  }

  /* send delayed reply to hello request */
  SendReplyToGetHello();

  /* save time to send hello */
  sendHelloTime = time(NULL);  

  /* set the hello wait mode ON */
  helloWait=TRUE;

  /* search new IPv6 addresses */
  ScanNdpEntry(alarmArg.pClientAddr, alarmArg.userid,
	     alarmArg.macAddr6, alarmArg.userProperty);

  /* mac address check */
  if(MacAddrCheck(alarmArg.ipStatus, 
		  alarmArg.clientAddr4, alarmArg.clientAddr6, 
		  alarmArg.macAddr4, alarmArg.macAddr6)==FALSE){
    connectMode=ENDCONNECT;
    return;
  }

  /* set the alarm for next periodic check */
  checkHttpAlarmRinged=FALSE;
  AddAlarm("CheckHttpAlarm", alarmArg.checkInterval, FALSE, OnCheckHttpAlarm);
  /* EnableAlarm is called automatically in alarm function */

  /* normal return */
  connectMode = HTTPCONNECT;
  return;
}

/***************************************************/
/* read skip Http headers ended by null line       */
/***************************************************/
void readHttpHeaders(void)
{
  char buff[BUFFMAXLN];             /* read in buffer */
  int n;

  /* read until null line (only CRLF code) */
  while((n=readln(connfd, buff, BUFFMAXLN))>0){
    ;
  }
}

/********************************************************/
/* send hello to client for replying GET /hello request */
/********************************************************/
void sendReplyToGetHello(void)
{
  /* send HTTP headers */
  Writefmt(connfd,"HTTP/1.1 200 OK\r\n");
  Writefmt(connfd,"Transfer-Encoding: chunked\r\n");
  Writefmt(connfd,"Keep-Alive: timeout=300\r\n");
  Writefmt(connfd,"Connection: Keep-Alive\r\n");
  Writefmt(connfd,"Content-Type: text/html\r\n");
  Writefmt(connfd,"\r\n");

  /* send reply to hello */
  Writefmt(connfd,"5\r\n");
  Writefmt(connfd,"hello\r\n"); 

  /* send end of chunk */
  Writefmt(connfd,"0\r\n");
  Writefmt(connfd,"\r\n");
}

/***************************************************/
/* send httpkeep page to client via connfd         */
/***************************************************/
void sendHttpKeepPage(char *userid, char *sessionId, char *language, int port, int cookieAuth, char *redirectedUrl)
{
  char buff[BUFFMAXLN];             /* read in buffer */
  FILE *fp;

  char httpKeepDoc[BUFFMAXLN];
  char httpHelloUrl[BUFFMAXLN];
  char terminateUrl[BUFFMAXLN];
  char acceptDoc2Url[BUFFMAXLN];
  char httpkeepJsUrl[BUFFMAXLN];
  char md5JsUrl[BUFFMAXLN];
  char portStr[WORDMAXLN];
  char *startPageUrl="";
  int startPageType=atoi(GetConfValue("StartPage/Type"));

  char *opengateServerName=GetConfValue("OpengateServerName");
  char *opengateDir=GetConfValue("OpengateDir");
  char useridshort[USERMAXLN];
  char extraId[USERMAXLN];

  /* split id to display short format of userid */
  SplitId(userid, useridshort, extraId);

  /* create path to accept2 doc */
  snprintf(acceptDoc2Url, BUFFMAXLN, 
	  "http://%s%s/%s/%s",opengateServerName,
	  opengateDir,language,GetConfValue("AcceptDoc2"));

  /* create terminate url [http://<servaddr>:<port>/terminate-<pid>] */
  snprintf(terminateUrl, BUFFMAXLN, "http://%s:%d/terminate-%d", 
	  opengateServerName, port, getpid());

  /* create httphello url [http://<servaddr>:<port>/hello] */
  snprintf(httpHelloUrl, BUFFMAXLN, "http://%s:%d/hello", 
	  opengateServerName, port);

  /* create httpkeep.js url[http://<serveraddr>/opengate/httpkeep.js] */
  snprintf(httpkeepJsUrl, BUFFMAXLN, "http://%s%s/%s", 
	   opengateServerName,
	   opengateDir,GetConfValue("HttpKeepJS"));

  /* create md5.js url[http://<serveraddr>/opengate/md5.js] */
  snprintf(md5JsUrl, BUFFMAXLN, "http://%s%s/%s", 
	   opengateServerName,
	   opengateDir,GetConfValue("Md5JS"));

  /* create path to httpkeep doc */
  snprintf(httpKeepDoc,BUFFMAXLN, "%s%s/%s/%s",GetConfValue("DocumentRoot"),
	  opengateDir,language,GetConfValue("HttpKeepDoc"));

  /* create port string */
  snprintf(portStr, WORDMAXLN, "%d", port);

  /* open httpkeepdoc */
  if((fp=fopen(httpKeepDoc, "r"))==NULL){
    err_msg("ERR at %s#%d: cannot open %s",__FILE__,__LINE__,httpKeepDoc);
    PutClientMsg("Cannot find html document");
    return;
  }

  /* if redirect page is not set, use other setting */
  if(isNull(redirectedUrl)){
    if(isNull(GetConfValue("StartPage/Url"))) redirectedUrl=acceptDoc2Url;
    else  redirectedUrl=GetConfValue("StartPage/Url");
  }

  /* create start page url to put information */
  if(cookieAuth==1) startPageUrl=redirectedUrl;
  else if(startPageType==0) startPageUrl=acceptDoc2Url;
  else if(startPageType==1) startPageUrl=GetConfValue("StartPage/Url");
  else if(startPageType==2) startPageUrl=redirectedUrl;
  else startPageUrl=acceptDoc2Url;

  /* send HTTP headers */
  Writefmt(connfd,"HTTP/1.1 200 OK\r\n");
  Writefmt(connfd,"Transfer-Encoding: chunked\r\n");
  Writefmt(connfd,"Keep-Alive: timeout=300\r\n");
  Writefmt(connfd,"Connection: Keep-Alive\r\n");
  Writefmt(connfd,"Content-Type: text/html\r\n");
  Writefmt(connfd,"\r\n");

  while(fgets(buff, BUFFMAXLN, fp)!=NULL){

    /* length check */
    if(strlen(buff)>=BUFFMAXLN-1){
      err_msg("ERR at %s#%d: too long line in %s",__FILE__,__LINE__,
	      httpKeepDoc);
    }

    /* replace mark */
    htmlReplace(buff, "%%OPENGATESERVERNAME%%", opengateServerName);
    htmlReplace(buff, "%%HTTPHELLOURL%%", httpHelloUrl);
    htmlReplace(buff, "%%USERID%%", useridshort);
    htmlReplace(buff, "%%SESSIONID%%", sessionId);
    htmlReplace(buff, "%%TERMINATEURL%%", terminateUrl);
    htmlReplace(buff, "%%HTTPKEEPJSURL%%", httpkeepJsUrl);
    htmlReplace(buff, "%%MD5JSURL%%", md5JsUrl);

    htmlReplace(buff, "%%OPENGATEDIR%%", opengateDir);
    htmlReplace(buff, "%%OPENGATEPORT%%", portStr);
    htmlReplace(buff, "%%LANGUAGE%%", language);
    htmlReplace(buff, "%%STARTURL%%", startPageUrl);
    htmlReplace(buff, "%%COOKIENAME%%", COOKIENAME);
    htmlReplace(buff, "%%REDIRECTEDURL%%", redirectedUrl);

    /* length of chunk in hex */
    Writefmt(connfd, "%x\r\n", strlen(buff));
    /* the chunk content */
    Writefmt(connfd, "%s\r\n", buff);
  }
  
  /* chunk end */
  Writefmt(connfd,"0\r\n");
  Writefmt(connfd,"\r\n");

  fclose(fp);
}


/*************************************************************/
/* send reply to unknown http request eg:[GET /favico.ico..] */
/*************************************************************/
void sendHttpReply(char *reply)
{
  /* send HTTP headers */
  Writefmt(connfd,"HTTP/1.1 200 OK\r\n");
  Writefmt(connfd,"Transfer-Encoding: chunked\r\n");
  Writefmt(connfd,"Keep-Alive: timeout=300\r\n");
  Writefmt(connfd,"Connection: Keep-Alive\r\n");
  Writefmt(connfd,"Content-Type: text/html\r\n");
  Writefmt(connfd,"\r\n");
  
  if(!isNull(reply)){
    /* length of chunk in hex */
    Writefmt(connfd, "%x\r\n", strlen(reply));
    /* the chunk content */
    Writefmt(connfd, "%s\r\n", reply);
  }

  /* chunk end */
  Writefmt(connfd,"0\r\n");
  Writefmt(connfd,"\r\n");
}

/*************************************************************/
/* send NOT FOUND reply to unknown http request              */
/*************************************************************/
void sendHttpNotFound(void)
{
  /* send HTTP headers */
  Writefmt(connfd,"HTTP/1.1 404\r\n");
  Writefmt(connfd,"\r\n");
}



/***************************************************/
/***************************************************/
void GetPeerAddr(int sockfd, char *peerAddr)
{
  if(debug>1) err_msg("DEBUG:=>getPeerAddr( )"); 
  getPeerAddr(sockfd,peerAddr);
  if(debug>1) err_msg("DEBUG:<=getPeerAddr(,%s)",peerAddr);
}


int GetListenPort(void)
{
  int ret;

  if(debug>1) err_msg("DEBUG:=>getListenPort( )");
  ret=getListenPort();
  if(debug>1) err_msg("DEBUG:(%d)<=getListenPort( )",ret);

  return ret;
}

int WaitClientConnect(char *userid, char *userProperty, char *sessionId, char *clientAddr4, char *clientAddr6, int duration, char *macAddr4, char *macAddr6, int ipStatus, struct clientAddr *pClientAddr, char *language, int port, int pid, int cookieAuth, char *redirectedUrl)
{
  int ret;

  if(debug>1) err_msg("DEBUG:=>waitClientConnect(%s,%s,%s,%s,%s,%d,%s,%s,%d,%s,%d,%d,%d,%s)",userid,userProperty,sessionId,clientAddr4,clientAddr6,duration,macAddr4,macAddr6,ipStatus,language,port,pid,cookieAuth,redirectedUrl);
  ret=waitClientConnect(userid,userProperty,sessionId,clientAddr4,clientAddr6,duration,macAddr4,macAddr6,ipStatus,pClientAddr,language,port,pid,cookieAuth,redirectedUrl);
  if(debug>1) err_msg("DEBUG:(%d)<=waitClientConnect( )",ret);

  return ret;
}

void SendQuitClient(void)
{
  if(debug>1) err_msg("DEBUG:=>sendQuitClient( )");
  sendQuitClient();
  if(debug>1) err_msg("DEBUG:<=sendQuitClient( )");
}

void SendTerminateReply(void)
{
  if(debug>1) err_msg("DEBUG:=>sendTerminateReply( )");
  sendTerminateReply();
  if(debug>1) err_msg("DEBUG:<=sendTerminateReply( )");
}

void WaitHttpClose(struct clientAddr *pClientAddr, char *userid, char *userProperty, char *macAddr4, char *macAddr6, int ipStatus, char *sessionid, int port)
{
  if(debug>1) err_msg("DEBUG:=>waitHttpClose(%p,%s,userProperty,%s,%s,%d,%s,%d)",pClientAddr,userid,macAddr4,macAddr6,ipStatus, sessionid, port);
  waitHttpClose(pClientAddr,userid,userProperty,macAddr4,macAddr6,ipStatus, sessionid, port);
  if(debug>1) err_msg("DEBUG:<=waitHttpClose( )");

}

void ReadHttpHeaders(void)
{
  if(debug>1) err_msg("DEBUG:=>readHttpHeaders( )");
  readHttpHeaders();
  if(debug>1) err_msg("DEBUG:<=readHttpHeaders( )");
}

void SendReplyToGetHello(void)
{
  if(debug>1) err_msg("DEBUG:=>sendReplyToGetHello( )");
  sendReplyToGetHello();
  if(debug>1) err_msg("DEBUG:<=sendReplyToGetHello( )");
}

void SendHttpKeepPage(char *userid, char *sessionId, char *language, int port, int cookieAuth, char *redirectedUrl)
{
  if(debug>1) err_msg("DEBUG:=>sendHttpKeepPage(%s,%s,%s,%d,%d,%s)", userid, sessionId, language, port, cookieAuth, redirectedUrl);
  sendHttpKeepPage(userid, sessionId, language, port, cookieAuth, redirectedUrl);
  if(debug>1) err_msg("DEBUG:<=sendHttpKeepPage( )");

}

void OnUsageTimeLimitAlarm(int signo){

  if(debug>1) err_msg("DEBUG:=>onUsageTimeLimitAlarm()");
  onUsageTimeLimitAlarm(signo);
  if(debug>1) err_msg("DEBUG:<=onUsageTimeLimitAlarm()");
}

void OnCheckBasicAlarm(int signo){

  if(debug>1) err_msg("DEBUG:=>onCheckBasicAlarm()");
  onCheckBasicAlarm(signo);
  if(debug>1) err_msg("DEBUG:<=onCheckBasicAlarm()");
}

void CheckBasic(void){

  if(debug>1) err_msg("DEBUG:=>checkBasic()");
  checkBasic();
  if(debug>1) err_msg("DEBUG:<=checkBasic()");
}

void OnCheckHttpAlarm(int signo){

  if(debug>1) err_msg("DEBUG:=>onCheckHttpAlarm()");
  onCheckHttpAlarm(signo);
  if(debug>1) err_msg("DEBUG:<=onCheckHttpAlarm()");
}

void CheckHttp(void){

  if(debug>1) err_msg("DEBUG:=>checkHttp()");
  checkHttp();
  if(debug>1) err_msg("DEBUG:<=checkHttp()");
}

void OnReadWaitAlarm(int signo){

  if(debug>1) err_msg("DEBUG:=>onReadWaitAlarm()");
  onReadWaitAlarm(signo);
  if(debug>1) err_msg("DEBUG:<=onReadWaitAlarm()");
}

void OnAjaxWaitAlarm(int signo){

  if(debug>1) err_msg("DEBUG:=>onAjaxWaitAlarm()");
  onAjaxWaitAlarm(signo);
  if(debug>1) err_msg("DEBUG:<=onAjaxWaitAlarm()");
}

int SelectAccept(void){
  int ret;

  if(debug>1) err_msg("DEBUG:=>selectAccept()");
  ret=selectAccept();
  if(debug>1) err_msg("DEBUG:(%d)<=selectAccept()",ret);

  return ret;
}

void AcceptHttpReConnect(int timeout){
  if(debug>1) err_msg("DEBUG:=>acceptHttpReConnect(%d)",timeout);
  acceptHttpReConnect(timeout);
  if(debug>1) err_msg("DEBUG:<=acceptHttpReConnect()");
}

int MacAddrCheck(int ipStatus, char *clientAddr4, char *clientAddr6, char *macAddr4, char *macAddr6) {
  int ret;

  if(debug>1) err_msg("DEBUG:=>macAddrCheck(%d,%s,%s,%s,%s)", 
		      ipStatus, clientAddr4, clientAddr6, macAddr4, macAddr6);
  ret=macAddrCheck(ipStatus, clientAddr4, clientAddr6, macAddr4, macAddr6);
  if(debug>1) err_msg("DEBUG:(%d)<=macAddrCheck()",ret);

  return ret;
}

int IsRightKey(char *pNowKey, char *sessionId){
  int ret;

  if(debug>1) err_msg("DEBUG:=>isRightKey(%s,%s)", pNowKey, sessionId);
  ret=isRightKey(pNowKey, sessionId);
  if(debug>1) err_msg("DEBUG:(%d)<=isRightKey()",ret);

  return ret;
}

void SendHttpReply(char *reply){
  if(debug>1) err_msg("DEBUG:=>sendHttpRepy(%s)", reply);
  sendHttpReply(reply);
  if(debug>1) err_msg("DEBUG:<=sendHttpReply()");
}


void SendHttpNotFound(void){
  if(debug>1) err_msg("DEBUG:=>sendHttpNotFound()");
  sendHttpNotFound();
  if(debug>1) err_msg("DEBUG:<=sendHttpNotFound()");
}

