/**************************************************
opengate server
 module for Controling ipfw 

Copyright (C) 1999 Opengate Project Team
Written by Yoshiaki Watanabe
Modified Katsuhiko Eguchi, 2005 

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Email: watanaby@is.saga-u.ac.jp
**************************************************/

#include "opengatesrv.h"

char ruleNumber4[WORDMAXLN];  /* ipfw rule number in string form */

int getRuleNumber4(char *clientAddr4);
int GetRuleNumber4(char *clientAddr4);

static void sigFunc(int signo);

/******************************************************************/
/* open gate for clientAddr4 (nnn.nnn.nnn.nnn)                     */
/******************************************************************/
int openClientGate4(char *clientAddr4, char *userid, char *macAddr4, char *userProperty)
{
  int fd;
  int ret=0;
  int retNum;

  Sigfunc *defaultSigFunc;

  /* exclusive exec of ipfw to avoid overlapped rule number */
  /**** prepare ****/
  /* open lockfile */
  fd=open(GetConfValue("LockFile"), O_RDWR|O_CREAT, 
	  S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
  if(fd==-1){
    err_msg("ERR at %s#%d: lockfile open error",__FILE__,__LINE__);
    return 1;
  } 

  /* set timeout */
  if((defaultSigFunc=Signal(SIGALRM, sigFunc))==SIG_ERR) return 1;
  alarm(atoi(GetConfValue("LockTimeout")));

  /* lock */
  if(Lock(fd)<0){
    err_msg("ERR at %s#%d: lock error",__FILE__,__LINE__);
    return 1;
  }

  /* reset timeout */
  Signal(SIGALRM, defaultSigFunc);
  alarm(0);

  /**** read rules ****/
  if((retNum=GetRuleNumber4(clientAddr4))<0){
    /* fail then unlock */
    Unlock(fd);
    Close(fd);
    return retNum;
  }

  /**** write rules ****/

  if(atoi(GetConfValue("IpfwScript/Enable"))){
    /********** use perl script to control firewall ************/

    if(Systeml(1, GetConfValue("IpfwScript/Path"),GetConfValue("IpfwPath"),
	       ruleNumber4,clientAddr4,
	       userid,macAddr4,userProperty,
	       GetConfValue("IpfwTagNumber"),(char *)0) != 0){
      err_msg("ERR at %s#%d: exec script error",__FILE__,__LINE__);
      ret=1;  /* abnormal */
    }
    
    /* lock is not necessary in following exec */
    Unlock(fd);
    Close(fd);    /* because reserved number is used */

  }else{
    /********** direct control of firewall **********************/
    /********** add outgoing ipfw rule for the client *************/
    if(Systeml(1, GetConfValue("IpfwPath"),"-q","add",ruleNumber4,
	       "count","tag",GetConfValue("IpfwTagNumber"),
	       "ip","from",clientAddr4,"to","any",(char *)0) != 0){
      err_msg("ERR at %s#%d: exec ipfw add error",__FILE__,__LINE__);
      ret=1;  /* abnormal */
    }

    /* lock is not necessary in following exec */
    Unlock(fd);
    Close(fd);    /* because reserved number is used */
    
    if(Systeml(1, GetConfValue("IpfwPath"),"-q","add",ruleNumber4,
	       "count","tag",GetConfValue("IpfwTagNumber"),
	       "ip","from","any","to",clientAddr4,(char *)0) != 0){
      err_msg("ERR at %s#%d: exec ipfw add error",__FILE__,__LINE__);
      ret=1; /* abnormal */
    }
  }
  return ret;
}


/******************************************************************/
/* close gate for clientAddr (nnn.nnn.nnn.nnn)                    */
/******************************************************************/
void closeClientGate4(struct clientAddr *pClientAddr, char *userid, char *macAddr4)
{
  double time_l;
  int hour, min, sec;
  time_t timeOut;

  /********** del ipfw rule for the client *************/
  DelIpfwRule(pClientAddr->ruleNumber);

  timeOut = time(NULL);
  time_l=difftime(timeOut,pClientAddr->timeIn);
  hour=time_l/60/60;
  min=(time_l-hour*60*60)/60;
  sec=(time_l-hour*60*60-min*60);
  err_msg("CLOS: user %s from %s at %s ( %02d:%02d:%02d )",
	  userid, pClientAddr->ipAddr, macAddr4, hour,min,sec);

  return;
}

/***********************************************/
/* delete ipfw rule                            */
/***********************************************/
void delIpfwRule(char *ruleNumber){
  int count;

  /* count rule */
  count=CountRuleNumber4(ruleNumber);

  if(count>0){
    /* exec ipfw del */
    /* [ipfw del rule] deletes all rule of the rule number at one call */
    if(Systeml(1, GetConfValue("IpfwPath"),"delete",ruleNumber,(char *)0) != 0){
      err_msg("ERR at %s#%d: exec ipfw del error",__FILE__,__LINE__);
    }
  }
}

/**************************************/
/* get unused ipfw rule number        */
/* error if addr is already in rules */ 
/* return value ret>0: acquired rule number that can be used */
/*              ret=-1: no rule number available */
/*              ret=-2: some system error occured */
/*              ret=-num: the ip address is already registered in rule 'num' */
/**************************************/
int getRuleNumber4(char *clientAddr4)
{
  FILE *fpipe;
  char buf[BUFFMAXLN];
  int num,newNum,readinNum;
  char *p;
  int ipfwmin;
  int ipfwmax;
  int ipfwinterval;
  int portStatus;
  int fileStatus;
  enum status {NORMAL, ABNORMAL, FOUND, NOTFOUND, DUPLICATED};

  /* exec ipfw list and open pipe */
  if((fpipe=Popenl(1, "r", GetConfValue("IpfwPath"),"list",(char *)0)) == NULL){ 
      err_msg("ERR at %s#%d: exec ipfw list error",__FILE__,__LINE__);
  }
  
  /* search unused rule number in the list read from pipe */
  /* check duplication of clientAddr to existing rules */

  newNum=-1;
  readinNum=0;
  portStatus=NOTFOUND;
  fileStatus=NORMAL;

  /* get rule range from config */
  ipfwmin=atoi(GetConfValue("IpfwRule/Min"));
  ipfwmax=atoi(GetConfValue("IpfwRule/Max"));
  ipfwinterval=atoi(GetConfValue("IpfwRule/Interval"));

  /* each port is checked whether it can be used for new rule or not */
  for(num=ipfwmin;num<=ipfwmax;num+=ipfwinterval){

    /* skip rules smaller than num */
    while(readinNum<num){
      if(fgets(buf, BUFFMAXLN, fpipe)==NULL){
	if(feof(fpipe)==1) fileStatus=EOF; 
	else fileStatus=ABNORMAL;
	break;
      }
      if( sscanf(buf, "%d", &readinNum) !=1 ){
	err_msg("ERR at %s#%d: abnormal ipfw response[ %s ]",
		__FILE__,__LINE__, buf);
	fileStatus=ABNORMAL; /* abnormal responsem exit internal loop */
	break;
      }
    }
    
    if(fileStatus==ABNORMAL){
      /* abnormal file proc, exit external loop */ 
      break;
    }

    if(fileStatus==EOF){
      /* EOF before reading a rule that is larger or equal to num */
      /* it means that num can be used for new client */
      portStatus=FOUND;
      newNum=num;
      break;
    }

    /* at this point, readinNum is larger or equal to num */
    /* check number duplication */
    if(readinNum==num){

      /* if clientAddr is found in the existing rule, then err exit. */
      if(((p=(char*)strstr(buf+1,clientAddr4))!=NULL)
	&& isspace(*(p-1))
	&& !isalnum(*(p+strlen(clientAddr4)))){
	err_msg("ERR at %s#%d: overlapped request from %s",
		__FILE__,__LINE__, clientAddr4);
	newNum=num;
	portStatus=DUPLICATED;
	break;
      }
      /* the num is used for other client */
      /* go to checking of next num */
      else{
	continue;
      }
    }
    
    /* at this point, readNum is larger than num */
    /* it means that num can be used for new client */
    newNum=num;
    portStatus=FOUND;
    break;
  }
  
  /* close pipe */
  Pclose(fpipe);
    
  if(fileStatus==ABNORMAL){
    err_msg("ERR at %s#%d: abnormal ipfw response ",__FILE__,__LINE__);
    return -2;
  }
  if(portStatus==NOTFOUND){
    err_msg("ERR at %s#%d: cannot get unused ipfw number",__FILE__,__LINE__);
    return -1;
  }
  if(portStatus==DUPLICATED){
    return -newNum;
  }

  snprintf(ruleNumber4, WORDMAXLN, "%d", newNum); /* to string */

  return newNum;
}

/*******************************/
/* get packet count from ipfw  */
/*******************************/
int getPacketCount4(char *ruleNumber)
{
  FILE *fpipe;
  char buf[BUFFMAXLN];
  int rule;
  int packets,packetsSum;

  /* exec proc */
  if((fpipe=Popenl(1, "r", GetConfValue("IpfwPath"),"-a","list",ruleNumber,(char *)0)) == NULL){ 
    err_msg("ERR at %s#%d: exec ipfw -a list error",__FILE__,__LINE__);
  }

  /* search unused number in the list read from pipe */
  packetsSum=0;
    
  while(fgets(buf, BUFFMAXLN, fpipe)!=NULL){
    sscanf(buf, "%d %d", &rule, &packets);   /* get packet count */
    packetsSum+=packets;
  }

  /* close pipe */
  Pclose(fpipe);

  return packetsSum;
}

/**********************************************/
/* get rule count registed to a rule number   */
/**********************************************/
int countRuleNumber4(char *ruleNumber)
{
  FILE *fpipe;
  char buf[BUFFMAXLN];
  int ruleCount;

  /* exec proc */
  if((fpipe=Popenl(1, "r", GetConfValue("IpfwPath"),"list",ruleNumber,(char *)0)) == NULL){ 
    err_msg("ERR at %s#%d: exec ipfw list error",__FILE__,__LINE__);
  }
  
  /* count line read from pipe */
  ruleCount = 0;
  while(fgets(buf, BUFFMAXLN, fpipe)!=0) ruleCount++;

  /* close pipe */
  Pclose(fpipe);

  return ruleCount;
}

/**********************************************/
/* function called by signal int              */
/**********************************************/
static void sigFunc(int signo)
{
  return;
}

/**********************************************/
/**********************************************/
int GetRuleNumber4(char *clientAddr4)
{
  int ret;

  if(debug>1) err_msg("DEBUG:=>getRuleNumber4(%s)",clientAddr4);
  ret=getRuleNumber4(clientAddr4);
  if(debug>1) err_msg("DEBUG:(%d)<=getRuleNumber4( )",ret);

  return ret;
}

int OpenClientGate4(char *clientAddr4, char *userid, char *macAddr4, char *userProperty)
{
  int ret;

  if(debug>1) err_msg("DEBUG:=>openClientGate4(%s,%s,%s,%s)",clientAddr4,userid,macAddr4,userProperty);
  ret=openClientGate4(clientAddr4, userid, macAddr4, userProperty);
  if(debug>1) err_msg("DEBUG:(%d)<=openClientGate4( )",ret);

  return ret;
}

void CloseClientGate4(struct clientAddr *pClientAddr, char *userid, char *macAddr4)
{
  if(debug>1) err_msg("DEBUG:=>closeClientGate4(%p,%s,%s)",pClientAddr,userid,macAddr4);
  closeClientGate4(pClientAddr,userid,macAddr4);
  if(debug>1) err_msg("DEBUG:<=closeClientGate4( )");
}


int GetPacketCount4(char *ruleNumber)
{
  int ret;

  if(debug>1) err_msg("DEBUG:=>getPacketCount4(%s)",ruleNumber);
  ret=getPacketCount4(ruleNumber);
  if(debug>1) err_msg("DEBUG:(%d)<=getPacketCount4( )",ret);

  return ret;
}

void DelIpfwRule(char *ruleNumber){
 if(debug>1) err_msg("DEBUG:=>delIpfwRule(%s)",ruleNumber);
  delIpfwRule(ruleNumber);
  if(debug>1) err_msg("DEBUG:<=delIpfwRule( )");
}

int CountRuleNumber4(char *ruleNumber)
{
  int ret;
  
  if(debug>1) err_msg("DEBUG:=>countRuleNumber4(%s)", ruleNumber);
  ret=countRuleNumber4(ruleNumber);
  if(debug>1) err_msg("DEBUG:(%d)<=countRuleNumber4( )",ret);
  
  return ret;
}
