/**************************************************
opengate server
 module for Authentication of User

Copyright (C) 1999 Opengate Project Team
Written by Yoshiaki Watanabe
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Email: watanaby@is.saga-u.ac.jp
**************************************************/

#include "opengatesrv.h"

void onAuthReplyAlarm(int signo);
int authFtp(char *userid, char *passwd);
int AuthFtp(char *userid, char *passwd);
int authPop3(char *userid, char *passwd);
int AuthPop3(char *userid, char *passwd);
int authPam(char *userid, char *passwd);
int AuthPam(char *userid, char *passwd);
int authRadius(char *userid, char *passwd);
int AuthRadius(char *userid, char *passwd);
int authPop3s(char *userid, char *passwd);
int AuthPop3s(char *userid, char *passwd);
int authFtpse(char *userid, char *passwd);
int AuthFtpse(char *userid, char *passwd);
int authFtpsi(char *userid, char *passwd);
int AuthFtpsi(char *userid, char *passwd);
int authLdap(char *userid, char *passwd);
int AuthLdap(char *userid, char *passwd);

/*******************************************************/
/* Authenticate user by accessing to ftp server        */
/*  userid : user to auth                              */
/*  password : password for the user                   */
/*******************************************************/
int authenticateUser(char *userid, char *passwd)
{
  char* proto;
  int authResult;
  int timeout;

  /* if no userid or no passwd, then deny */
  //if(userid[0]=='\0')return DENY;
  //if(passwd[0]=='\0')return DENY;

  /* authserver get timeout value */
  timeout=atoi(GetConfValue("AuthServer/Timeout"));

  /* set auth server reply timeout */
  if(timeout>0){
    AddAlarm("AuthReplyAlarm", timeout, TRUE, onAuthReplyAlarm); 
    EnableAlarm();
  }

  /* get Protocol for authentication */
  proto=GetConfValue("AuthServer/Protocol");

  if(strcmp(proto, "ftp")==0){
    /* authenticate by ftp access */
    authResult=AuthFtp(userid, passwd);

  }else if(strcmp(proto, "pop3")==0){
    /* authenticate by pop3 access */
    authResult=AuthPop3(userid, passwd);

  }else if(strcmp(proto, "pam")==0){
    /* authenticate by pam */
    authResult=AuthPam(userid, passwd);

  }else if(strcmp(proto, "radius")==0){
    /* authenticate by radius */
    authResult=AuthRadius(userid, passwd);

  }else if(strcmp(proto, "pop3s")==0){
    /* authenticate by pop3s */
    authResult=AuthPop3s(userid, passwd);

  }else if(strcmp(proto, "ldap")==0){
    /* authenticate by ldap */
    authResult=AuthLdap(userid, passwd);

  }else if(strcmp(proto, "ftpse")==0){
    /* authenticate by ftps-explicit */
    authResult=AuthFtpse(userid, passwd);

  }else if(strcmp(proto, "ftpsi")==0){
    /* authenticate by ftps-implicit */
    authResult=AuthFtpsi(userid, passwd);

  }else if(strcmp(proto, "deny")==0){
    /* deny all request */
    authResult=DENY;

  }else if(strcmp(proto, "accept")==0){
    /* accept all request */
    authResult=ACCEPT;

  }else{
    err_msg("ERR at %s#%d: Unknown protocol:%s",__FILE__,__LINE__,proto);
    authResult=DENY;
  }

  /* stop the auth reply timeout */
  if(timeout>0) RemoveAlarm("AuthReplyAlarm");

  return authResult;
}

/*****************************/
/* split userid and extraId */
/*****************************/
void splitId(char* userid, char* useridshort, char* extraId)
{
  char useridSave[USERMAXLN];

  char* markPnt=NULL;

  strncpy(useridSave, userid, USERMAXLN);

  /* separate extraId from userid */
  markPnt=strchr(useridSave, *GetConfValue("UserIdSeparator"));
  if(markPnt==NULL){  
    /* separator mark not found */
    strncpy(extraId,"default",USERMAXLN);
    strncpy(useridshort,useridSave,USERMAXLN);
  }else{
    /* pick up extraId */
    *markPnt='\0';
    strncpy(extraId,markPnt+1,USERMAXLN);
    strncpy(useridshort,useridSave,USERMAXLN);
  }
} 

/***********************/
/* Authenticate by FTP */
/***********************/
int authFtp(char *userid, char *passwd)
{
  int		sockfd, n;
  char		recvline[BUFFMAXLN];
  int           authResult;
  char* serverAddr; /* auth server address */
  char* port;      /* auth server port */

  /* get auth server address */
  serverAddr=GetConfValue("AuthServer/Address");

  if(isNull(serverAddr)){
    err_msg("ERR at %s#%d: Missing address for FTP server in config",
	    __FILE__,__LINE__);
    return DENY;
  }

  /* get auth server port */
  port=GetConfValue("AuthServer/Port");

  /* FTP server connect */
  if(isNull(port)){
    sockfd = Tcp_connect(serverAddr, "ftp");
  }else{
    sockfd = Tcp_connect(serverAddr, port);
  }
  if(sockfd<0){
    err_msg("ERR at %s#%d: Ftp server is not normal 0",__FILE__,__LINE__);
    return DENY;
  }

  /* get [220 <host> FTP server ..]*/
  if((n = readln(sockfd, recvline, BUFFMAXLN)) < 0) {
    err_msg("ERR at %s#%d: Ftp server is not normal 1",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }
  if(strstr(recvline,"220")!=recvline){
    err_msg("ERR at %s#%d: Ftp server is not normal 2",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }

  /* put [user <userid>] */
  Writefmt(sockfd, "user %s\r\n", userid);

  /* get [331 Password required ..] */
  if((n = readln(sockfd, recvline, BUFFMAXLN)) < 0) {
    err_msg("ERR at %s#%d: Ftp server is not normal 3",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }

  /* if multi-line greeting [220 ...] exist, skip them. */
  while(strstr(recvline,"220")==recvline){
    if((n = readln(sockfd, recvline, BUFFMAXLN)) < 0) {
      err_msg("ERR at %s#%d: Ftp server is not normal 3",__FILE__,__LINE__);
      Close(sockfd);
      return DENY;
    }
  }

  /* check [331 Password required ..] */
  if(strstr(recvline,"331")!=recvline){
    err_msg("ERR at %s#%d: Ftp server is not normal 4",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }

  /* put [pass <password>] */
  Writefmt(sockfd, "pass %s\r\n", passwd);

  /* get [230 User <userid> logged in] */
  if((n = readln(sockfd, recvline, BUFFMAXLN)) < 0) {
    err_msg("ERR at %s#%d: Ftp server is not normal 5",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }
  if(strstr(recvline,"230")==recvline){
    authResult=ACCEPT;
  }else{
    authResult=DENY;
  }
  
  /* put [quit] */
  Writefmt(sockfd,"quit\r\n");

  Close(sockfd);
  return authResult;
}

/************************/
/* Authenticate by POP3 */
/************************/
int authPop3(char *userid, char *passwd)
{
  int		sockfd, n;
  char		recvline[BUFFMAXLN];
  int           authResult;
  char*  serverAddr;
  char* port;

  /* get auth server address */
  serverAddr=GetConfValue("AuthServer/Address");

  if(isNull(serverAddr)){
    err_msg("ERR at %s#%d: Missing address for POP3 server in config",
	    __FILE__,__LINE__);
    return DENY;
  }

  /* get auth server port */
  port=GetConfValue("AuthServer/Port");

  /* POP3 server connect */
  if(isNull(port)){
    sockfd = Tcp_connect(serverAddr, "pop3");
  }else{
    sockfd = Tcp_connect(serverAddr, port);
  }
  if(sockfd<0){
    err_msg("ERR at %s#%d: Pop3 server is not normal 0",__FILE__,__LINE__);
    return DENY;
  }

  /* get [+OK POP3 <host> <ver> server ready]*/
  if((n = readln(sockfd, recvline, BUFFMAXLN)) < 0) {
    err_msg("ERR at %s#%d: Pop3 server is not normal 1",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }
  if(strstr(recvline,"+OK")!=recvline){
    err_msg("ERR at %s#%d: Pop3 server is not normal 2",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }

  /* put [user <userid>] */
  Writefmt(sockfd, "user %s\r\n", userid);

  /* get [+OK User name accepted, password please] */
  if((n = readln(sockfd, recvline, BUFFMAXLN)) < 0) {
    err_msg("ERR at %s#%d: Pop3 server is not normal 3",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }
  if(strstr(recvline,"+OK")!=recvline){
    err_msg("ERR at %s#%d: Pop3 server is not normal 4",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }

  /* put [pass <password>] */
  Writefmt(sockfd, "pass %s\r\n", passwd);

  /* get [+OK Mailbox open, <count> messages] */
  if((n = readln(sockfd, recvline, BUFFMAXLN)) < 0) {
    err_msg("ERR at %s#%d: Pop3 server is not normal 5",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }
  if(strstr(recvline,"+OK")==recvline){
    authResult=ACCEPT;
  }else{
    authResult=DENY;
  }
  
  /* put [quit] */
  Writefmt(sockfd,"quit\r\n");

  Close(sockfd);
  return authResult;
}

/***************************************/
/* called at auth reply timeout        */
/***************************************/
void onAuthReplyAlarm(int signo)
{
}

/**********************************/
/**********************************/

int AuthFtp(char *userid, char *passwd)
{
  int ret;

  if(debug>1) err_msg("DEBUG:=>authFtp(%s,passwd)",userid);
  ret=authFtp(userid,passwd);
  if(debug>1) err_msg("DEBUG:(%d)<=authFtp( )",ret);

  return ret;
}


int AuthPop3(char *userid, char *passwd)
{
  int ret;

  if(debug>1) err_msg("DEBUG:=>authPop3(%s,passwd)",userid);
  ret=authPop3(userid,passwd);
  if(debug>1) err_msg("DEBUG:(%d)<=authPop3( )",ret);

  return ret;
}

int AuthenticateUser(char *userid, char *password)
{
  int ret;

  if(debug>1) err_msg("DEBUG:=>authenticateUser(%s,passwd)",userid);
  ret=authenticateUser(userid,password);
  if(debug>1) err_msg("DEBUG:(%d)<=authenticateUser( )",ret);

  return ret;
}

void SplitId(char* userid, char* useridshort, char* extraId)
{
  if(debug>1) err_msg("DEBUG:=>splitId(%s,,)",userid);
  splitId(userid,useridshort,extraId);
  if(debug>1) err_msg("DEBUG:<=splitId(%s,%s,%s)",userid,useridshort,extraId);
}
