/**************************************************
opengate server

module to control both of ipfw and ip6fw 

Copyright (C) 1999 Opengate Project Team
Written by Katsuhiko Eguchi, 2005 
Modified by Yoshiaki Watanabe

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Email: watanaby@is.saga-u.ac.jp
**************************************************/
#include	"opengatesrv.h"

extern char ruleNumber4[WORDMAXLN];  /* ipfw rule number in string form  */
extern char ruleNumber6[WORDMAXLN];  /* ip6fw rule number in string form */
struct clientAddr *pClientAddr = NULL;
int ipStatus;

void RemoveOverlapRule(int overlapRule4, int overlapRule6);
void KillOverlapProcess(int overlapRule4, int overlapRule6, 
			char *clientAddr4, char *clientAddr6);

/**************************************/
/* Open client gate for IPv4 and IPv6 */
/**************************************/
int openClientGate(char *clientAddr4, char *macAddr4, char *clientAddr6, char *macAddr6, char *userid, char *userProperty, int ipStatus)
{
  int ret;
  struct clientAddr *pLastClientAddr=NULL;
  int overlapRule4=0, overlapRule6=0;

  switch(ipStatus){

  /**** if client have IPv6 & IPv4 addresses, do below *******/
  case IPV46DUAL:
    
    /*** at first, open IPv4 gate *****/
    if((ret=OpenClientGate4(clientAddr4, userid, macAddr4, userProperty))==0){

      /* open success */
      err_msg("OPEN: user %s from %s at %s", userid, clientAddr4, macAddr4);

      /* create new address list. head and tail pointer point the same item. */
      pLastClientAddr=pClientAddr=CreateAddrListItem(clientAddr4,ruleNumber4,IPV4);
    }

    /* open fail */
    else if(ret==-2 || ret==-1 || ret>0){
      PutClientMsg("Error: Please contact to the administrator");
      return FALSE;
    }

    /* other case (ret<-2) means overlapped rule */
    else overlapRule4=-ret;


    /*** next, open IPv6 gate *****/
    if((ret=OpenClientGate6(clientAddr6, userid, macAddr6, userProperty))==0){

      /* open success */
      err_msg("OPEN: user %s from %s at %s", userid, clientAddr6, macAddr6);

      /* add the address info to list */
      pLastClientAddr->next = CreateAddrListItem(clientAddr6,ruleNumber6,IPV6);
    }
    /* open fail */
    else{
      /* remove previously registered ipv4 rule from list */ 
      if(pClientAddr!=NULL && pClientAddr->ipType==IPV4){
	CloseClientGate4(pClientAddr,userid,macAddr4);
	free(pClientAddr);
	pClientAddr=NULL;
      }
      /* some system error */
      if(ret==-2 || ret==-1 || ret>0){
	PutClientMsg("Error: Please contact to the administrator");
	return FALSE;
      }
      
      /* other case (ret<-2) means overlapped rule */
      else overlapRule6=-ret;
    }
      break;  /* case IPV46DUAL end */


    /***** if client have IPv4 only, do below *******/
  case IPV4ONLY:

    /* open IPv4 gate */
    if((ret=OpenClientGate4(clientAddr4, userid, macAddr4, userProperty))==0){

      /* open success */
      err_msg("OPEN: user %s from %s at %s", userid, clientAddr4, macAddr4);

      /* create new address list. head and tail pointer point the same item. */
      pLastClientAddr=pClientAddr
	=CreateAddrListItem(clientAddr4,ruleNumber4,IPV4);
    }

    /* open fail */
    else if(ret==-2 || ret==-1 || ret>0){
      PutClientMsg("Error: Please contact to the administrator");
      return FALSE;
    }

    /* other case (ret<-2) means overlapped rule */
    else overlapRule4=-ret;

    break; /* case IPV4ONLY end */
    

    /***** if client have IPv6 only, do below ******/
  case IPV6ONLY:
    /* open IPv6 gate */
    if((ret=OpenClientGate6(clientAddr6, userid, macAddr6, userProperty))==0){

      /* open success */
      err_msg("OPEN: user %s from %s at %s", userid, clientAddr6, macAddr6);

      /* create new address list. head and tail pointer point the same item. */
       pLastClientAddr=pClientAddr
	 =CreateAddrListItem(clientAddr6,ruleNumber6,IPV6);
     }

    /* open fail */
    else if(ret==-2 || ret==-1 || ret>0){
      PutClientMsg("Error: Please contact to the administrator");
      return FALSE;
    }
    
    /* other case (ret<-2) means overlapped rule */
    else overlapRule6=-ret;

    break; /* case IPv6ONLY end */

  default:
    err_msg("ERR at %s#%d: abnormal IP versions %d",__FILE__,__LINE__,ipStatus);

  } /* switch end */
  

  /* if overlapped rules exist, remove them */
  if(overlapRule4>0 || overlapRule6>0){

    /* to cope to modification of IPv4 addr in Dual stack(send as http param)*/ 
    if(ipStatus==IPV46DUAL && IsSameMacAddr(macAddr4, macAddr6)==FALSE){
      err_msg("ERR at %s#%d: arp and ndp return different mac addr",
	      __FILE__,__LINE__);
      PutClientMsg("Error. Please End Web and Retry. ");
    
    }else{

      /* ipfw rule is deleted and the process is killed */
      RemoveOverlapRule(overlapRule4,overlapRule6);
      KillOverlapProcess(overlapRule4,overlapRule6,clientAddr4,clientAddr6);

      err_msg("INFO: user %s forces to close overlapped client", userid);
      PutClientMsg("Network is closed. Please End Web and Retry. ");
    }
    return FALSE;
  }

  return TRUE;
}

/*************************************************/
/* remove overlapped firewall rule for same addr */
/*************************************************/
void removeOverlapRule(int overlapRule4, int overlapRule6)
{
  char ruleNumStr[WORDMAXLN];

  /* if ipfw rule exist, delete it */
  if(overlapRule4>0){
    snprintf(ruleNumStr,WORDMAXLN,"%d",overlapRule4);
    DelIpfwRule(ruleNumStr);
  }

  /* if ip6fw rule exist, delete it */
  if(overlapRule6>0){
    snprintf(ruleNumStr,WORDMAXLN,"%d",overlapRule6);
    DelIp6fwRule(ruleNumStr);
  }

}

/*************************************************/
/* kill overlapped process                       */
/*************************************************/
void killOverlapProcess(int overlapRule4, int overlapRule6, char *clientAddr4, char *clientAddr6){

  FILE *fpipe;
  char buff[BUFFMAXLN];
  char cgiNamePattern[WORDMAXLN];
  char addr4Pattern[ADDRMAXLN]="*dummy*";
  char addr6Pattern[ADDRMAXLN]="*dummy*";
  int procNumber=0;
  char procNumStr[WORDMAXLN];

  /* make cgi name pattern in ps title */
  snprintf(cgiNamePattern,WORDMAXLN," %s: ", GetConfValue("MainCgi"));
  
  /* make cgi param pattern in ps title */
  if(overlapRule4>0){
    snprintf(addr4Pattern,ADDRMAXLN,"[%s(%d)]", clientAddr4, overlapRule4);
  }
  if(overlapRule6>0){
    snprintf(addr6Pattern,ADDRMAXLN,"[%s(%d)]", clientAddr6, overlapRule6);
  }
 
  /* search process corresponding to the overlap rule */
  /* opengate.cgi process shows the following title line for ps command */
  /* "123 ... opengatesrv.cgi: ..,[123.4.5.6(10000)],[2001:1:1:1::1(10000)]..." */

  /* exec "ps -axww" */
  if((fpipe=Popenl(1, "r", GetConfValue("PsPath"), "-axww", (char *)0)) == NULL){ 
    err_msg("ERR at %s#%d: exec ps ax error",__FILE__,__LINE__);
  }
    
  /* read "ps -axww" output as follows */
  while(fgets(buff, BUFFMAXLN, fpipe)!=NULL){

    /* if opengatesrv cgi corresponding to overlap rule */
    if(strstr(buff, cgiNamePattern)!=NULL &&
       ( strstr(buff, addr4Pattern)!=NULL ||
	 strstr(buff, addr6Pattern)!=NULL ) ){

      /* get the process number */
      sscanf(buff, "%d", &procNumber);

      /* kill the process */
      snprintf(procNumStr, WORDMAXLN, "%d", procNumber);
      if(Systeml(1, "kill", procNumStr, (char *)0) != 0){
	err_msg("ERR at %s#%d: exec kill error",__FILE__,__LINE__);
      }
    }
  }
	   
  /* close pipe */
  Pclose(fpipe);
}  

/**************************/
/* make address list      */
/**************************/
struct clientAddr *createAddrListItem(char* ipAddr, char *ruleNumber, int ipType)
{
  struct clientAddr *pNew;

  if((pNew = (struct clientAddr *) malloc(sizeof(struct clientAddr))) == NULL){
    err_msg("ERR at %s#%d: malloc error",__FILE__,__LINE__);
    PutClientMsg("Error: Please contact to the administrator");
    exit(1);
  }
  strncpy(pNew->ipAddr,ipAddr,ADDRMAXLN);
  strncpy(pNew->ruleNumber,ruleNumber,WORDMAXLN);
  pNew->ipType = ipType;
  pNew->timeIn = time(NULL);
  pNew->activeStatus=TRUE;
  pNew->next = NULL;
  return pNew;
}

/*****************************/
/* Check enable IP versions  */
/*****************************/
int checkIpVersions(char *clientAddr4, char *clientAddr6){

  if(strstr(clientAddr4,".")!=NULL){
    if(strstr(clientAddr6,":")!=NULL){
      ipStatus = IPV46DUAL;
    }else{
      ipStatus = IPV4ONLY;
    }
  }else{
    if(strstr(clientAddr6,":")!=NULL){
      ipStatus = IPV6ONLY;
    }else{
      err_msg("ERR at %s#%d: Can not recognize IP versions for v4:%s v6:%s"
	      ,__FILE__,__LINE__,clientAddr4,clientAddr6);
      ipStatus=0;
    }
  }
  return ipStatus;
}

/****************************************/
/* get packet count from ipfw and ip6fw */
/****************************************/
int getPacketCount(struct clientAddr *pClientAddr)
{
  int packets=0;
  struct clientAddr *pNow;
  pNow=pClientAddr;

  while(pNow!=NULL){
    if(pNow->ipType==IPV4){
      packets+=GetPacketCount4(pNow->ruleNumber);
    }else if(pNow->ipType==IPV6){
      packets+=GetPacketCount6(pNow->ruleNumber);
    }
    pNow=pNow->next;
  }

  return packets;
}

/************************************/
/************************************/

int OpenClientGate(char *clientAddr4, char *macAddr4, char *clientAddr6, char *macAddr6, char *userid, char *userProperty, int ipStatus)
{
  int ret;
  if(debug>1) err_msg("DEBUG:=>openClientGate(%s,%s,%s,%s,%s,%s,%d)",clientAddr4,macAddr4,clientAddr6,macAddr6,userid,userProperty, ipStatus);
  ret = openClientGate(clientAddr4,macAddr4,clientAddr6,macAddr6,userid,userProperty, ipStatus);
  if(debug>1) err_msg("DEBUG:(%d)<=openClientGate()",ret);
  return ret;
}

struct clientAddr *CreateAddrListItem(char* ipAddr, char *ruleNumber, int ipType)
{
  struct clientAddr *ret;
  if(debug>1) err_msg("DEBUG:=>createAddrListItem(%s,%s,%d)",ipAddr,ruleNumber,ipType);
  ret = createAddrListItem(ipAddr, ruleNumber, ipType);
  if(debug>1) err_msg("DEBUG: [%p] <=createAddrListItem( )",ret);
  return ret;
}

int CheckIpVersions(char *clientAddr4, char *clientAddr6)
{
  int ret;
  if(debug>1) err_msg("DEBUG:=>checkIpVersions(%s,%s)", clientAddr4, clientAddr6);
  ret=checkIpVersions(clientAddr4, clientAddr6);
  if(debug>1) err_msg("DEBUG:(%d)<=checkIpVersions( )", ret);

  return ret;
}

void RemoveOverlapRule(int overlapRule4, int overlapRule6){

  if(debug>1) err_msg("DEBUG:=>removeOverlapRule(%d,%d)",overlapRule4,overlapRule6);
  removeOverlapRule(overlapRule4,overlapRule6);
  if(debug>1) err_msg("DEBUG:<=removeOverlapRule( )");
}

void KillOverlapProcess(int overlapRule4, int overlapRule6, 
		       char *clientAddr4, char *clientAddr6){
  if(debug>1) err_msg("DEBUG:=>killOverlapProcess(%d,%d,%s,%s)", 
		    overlapRule4,overlapRule6,clientAddr4,clientAddr6);
  killOverlapProcess(overlapRule4,overlapRule6,clientAddr4,clientAddr6);
  if(debug>1) err_msg("DEBUG:<=killOverlapProcess( )");
}


int GetPacketCount(struct clientAddr *pClientAddr)
{
  int ret;

  if(debug>1) err_msg("DEBUG:=>getPacketCount(%p)",pClientAddr);
  ret=getPacketCount(pClientAddr);
  if(debug>1) err_msg("DEBUG:(%d)<=getPacketCount( )",ret);

  return ret;
}
