/**************************************************
opengate server
 utility routines

Copyright (C) 1999 Opengate Project Team
Written by Yoshiaki Watanabe
Modified Katsuhiko Eguchi, 2005 

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Email: watanaby@is.saga-u.ac.jp
**************************************************/

#include "opengatesrv.h"

/*************************************************/
/* formated write                                */
/* fd : file descriptor                          */
/* fmt : format to write                         */
/* ... : terms to write                          */
/*************************************************/
void Writefmt(int fd, const char *fmt, ...)
{
  char	buff[BUFFMAXLN];
  va_list     ap;
  int nchar, nwrt;

  va_start(ap, fmt);
  vsnprintf(buff, BUFFMAXLN, fmt, ap);	
  va_end(ap);

  nchar=strlen(buff);
  nwrt=write(fd, buff, nchar);

  return;
}


void WritefmtSSL(SSL *fd, const char *fmt, ...)
{
  char	buff[BUFFMAXLN];
  va_list     ap;
  int nchar, nwrt;

  va_start(ap, fmt);
  vsnprintf(buff, BUFFMAXLN, fmt, ap);	
  va_end(ap);

  nchar=strlen(buff);
  nwrt=SSL_write(fd, buff, nchar);

  return;
}


/******************************************************/
/* Read one line                                      */
/*   fd: file descriptor                              */
/*   vptr: input buffer pointer                       */
/*   maxlen: buffer length                            */
/*                                                    */
/* the chars terminated with EOL or EOF is read in    */
/* ## this function assumes two EOL chars [CR LF]     */ 
/*  CRLF is not read in and skipped                   */
/* [abcdCRLFefghCRLF] => read[abcd],left[efghCRLF]    */
/*                                                    */
/* return value                                       */
/*    plus value means the count of chars to read     */
/*    value 0 means NULL line (no-chars & CRLF)       */
/*    value -1 means error (errno is set)             */
/*    value -2 means EOF (no-chars & EOF)             */ 
/******************************************************/
ssize_t
readln(int fd, void *vptr, size_t maxlen)
{
  ssize_t	n, rc;
  char	*ptr,c;

  ptr=vptr;

  /* pre read */
  rc = read(fd, &c, 1);
  if(rc==0) return(-2); /* EOF */
  if(rc<0) return(-1);  /* ERR */

  /* get char loop */
  n=0;
  while(n < maxlen-1) {
    if ( rc == 1) {      /* get some char */
      if (iscntrl(c)){         /* get control char (means EOL) */
	rc = read(fd, &c, 1);  /* skip second EOL char */
	break;
      }
      *ptr++ = c;
      n++;
    }else if (rc == 0) { /* EOF (but some chars are read already) */
      break;
    } else {             /* ERR */
      return(-1);
    }
    rc = read(fd, &c, 1);
  }
  /* null terminate string */  
  *ptr++ = 0;
  return(n);
}


ssize_t
readlnSSL(SSL *fd, void *vptr, size_t maxlen)
{
  ssize_t	n, rc;
  char	*ptr,c;

  ptr=vptr;

  /* pre read */
  rc = SSL_read(fd, &c, 1);
  if(rc==0) return(-2); /* EOF */
  if(rc<0) return(-1);  /* ERR */

  /* get char loop */
  n=0;
  while(n < maxlen-1) {
    if ( rc == 1) {      /* get some char */
      if (iscntrl(c)){      /* get control char (means EOL) */
	  rc = SSL_read(fd, &c, 1); /* skip second EOL char */
	  break;
      }
      *ptr++ = c;
      n++;
    }else if (rc == 0) { /* EOF (but some char are read already */
      break;
    } else {             /* ERR */
      return(-1);
    }
    rc = SSL_read(fd, &c, 1);
  }
  /* null terminate string */  
  *ptr++ = 0;
  return(n);
}


/******************************/
/* lock functions using fcntl */
/******************************/
int lock(int fd)
{
  struct flock lck;
  
  lck.l_type=F_WRLCK;
  lck.l_whence=SEEK_SET;
  lck.l_start=0;
  lck.l_len=0;
  return fcntl(fd, F_SETLKW, &lck);
}
  
/********************************/
/* unlock functions using fcntl */
/********************************/
int unlock(int fd)
{
  struct flock lck;

  lck.l_type=F_UNLCK;
  lck.l_whence=SEEK_SET;
  lck.l_start=0;
  lck.l_len=0;
  return fcntl(fd, F_SETLK, &lck);
}

/**************************************/
/* check NULL or point to null string */
/**************************************/
int isNull(const char *pStr)
{
  if(pStr==NULL) return TRUE;
  if(*pStr=='\0') return TRUE;
  return FALSE;
}



/**************************************************/
/* popen with argument list                       */
/* type : open type "r" or "w"                    */
/* path : command path to fork/exec               */
/* ... : command arguments. last must be (char*)0 */
/*  DO NOT SET user entered string in args        */
/**************************************************/
FILE *Popenl(const char *type, const char *path, ...)
{
  char	commandLine[BUFFMAXLN];
  va_list     ap;
  char *pStr;
  FILE *file;

  /* insert command path */
  strncpy(commandLine, path, BUFFMAXLN);

  /* insert command arguments */
  va_start(ap, path);
  
  while((pStr=va_arg(ap, char *))!=(char *)0){
    strcat(commandLine, " ");
    strncat(commandLine, pStr, BUFFMAXLN);
  }

  va_end(ap);

  /* open the pipe to the program  */
  if(debug>1) err_msg("DEBUG:=>popen(%s, %s)", commandLine, type);
  file=popen(commandLine, type);
  if(debug>1) err_msg("DEBUG:(%x)<=popen( )",file);  

  return file;
}


/**************************************************/
/* system with argument list                      */
/* path : command path to fork/exec               */
/* ... : command arguments. last must be (char*)0 */
/*  DO NOT SET user entered string in args        */
/**************************************************/
int Systeml(const char *path, ...)
{
  char	commandLine[BUFFMAXLN];
  va_list     ap;
  char *pStr;
  int ret;

  /* insert command path */
  strncpy(commandLine, path, BUFFMAXLN);

  /* insert command arguments */
  va_start(ap, path);
  
  while((pStr=va_arg(ap, char *))!=(char *)0){
    strcat(commandLine, " ");
    strncat(commandLine, pStr, BUFFMAXLN);
  }

  va_end(ap);

  /* execute shell  */
  if(debug>1) err_msg("DEBUG:=>system(%s)", commandLine);
  ret=system(commandLine);
  if(debug>1) err_msg("DEBUG:<=system()");

  return ret;
}

/*******************************************/
/* get port number string in /etc/services */
/*******************************************/
char *getServicePortStr(char *servName)
{
  struct servent *pServEnt;
  static char portStr[WORDMAXLN];

  /* get service info from service name */
  pServEnt = getservbyname(servName, NULL);

  if(pServEnt==NULL){
    err_msg("ERR at %s#%d: cannot find /etc/services entry for %s",
	    __FILE__,__LINE__,servName);
    return "";
  }

  /* convert service port number to string form */
  snprintf(portStr, sizeof(portStr),"%d",ntohs(pServEnt->s_port));

  return portStr;
}

/*******************************************/
/* create random session id                */
/*  simple but might be overlapped         */
/*  change logic, if you need identical id */ 
/*******************************************/
void createSessionId(char *sessionId)
{
  srandom(getpid()+time(NULL));
  snprintf(sessionId, BUFFMAXLN, "%ld", random() );
}
/*************************************************/
/* calc MD5 in hex form                          */
/*  str: plain text to convert                   */
/*  hexdigest: converted hex string              */
/*      prepare buff more or equal to 33chars    */
/*  len: length of hexdigest buffer              */
/*************************************************/
char *md5hex(char *hexdigest, int len, char *str)
{
  char unsigned digest[16];
  char hexcode[16]="0123456789abcdef";
  int i;
  
  /* if not enough buffer, exit */
  if(len<33){
    *hexdigest='\0';
    return hexdigest;
  }

  /* calc MD5 digest */
  MD5(str, strlen(str), digest);

  /* convert to HEX string */
  for(i=0;i<16;i++){
    hexdigest[2*i]=hexcode[digest[i]/16];
    hexdigest[2*i+1]=hexcode[digest[i]%16];
  }
  hexdigest[2*16]='\0';

  return hexdigest;
}

/****************************************/
/****************************************/
int Pclose(FILE *stream)
{
  int ret;

  if(debug>1) err_msg("DEBUG:=>pclose( )");
  ret = pclose(stream);
  if(debug>1) err_msg("DEBUG:<=pclose( )");  

  return ret;
}

char *GetServicePortStr(char *servName)
{
  char *ret;

  if(debug>1) err_msg("DEBUG:=>getServicePortStr(%s)", servName);
  ret = getServicePortStr(servName);
  if(debug>1) err_msg("DEBUG:(%s)<=getServicePortStr( )", ret);  

  return ret;
}

ssize_t Readln(int fd, void *ptr, size_t maxlen)
{
  ssize_t		n;

  if(debug>1) err_msg("DEBUG:=>readln( )");
  if ( (n = readln(fd, ptr, maxlen)) < 0){
    err_msg("ERR at %s#%d: readln error",__FILE__,__LINE__);
  }
  if(debug>1) err_msg("DEBUG:(%d)<=readln( )",n);

  return(n);
}

int Lock(int fd)
{
  int ret;

  if(debug>1) err_msg("DEBUG:=>lock( )");
  ret=lock(fd);
  if(debug>1) err_msg("DEBUG:(%d)<=lock( )",ret);

  return ret;
}


int Unlock(int fd)
{
  int ret;

  if(debug>1) err_msg("DEBUG:=>unlock( )");
  ret=unlock(fd);
  if(debug>1) err_msg("DEBUG:(%d)<=unlock( )",ret);

  return ret;
}


void CreateSessionId(char *sessionId){
  if(debug>1) err_msg("DEBUG:=>createSessionId( )");
  createSessionId(sessionId);
  if(debug>1) err_msg("DEBUG:<=createSessionId(%s)",sessionId);
}
