/**************************************************
opengate server
 module for getting parameters from conf file

Copyright (C) 2006 Opengate Project Team
Written by Yoshiaki Watanabe


This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Email: watanaby@is.saga-u.ac.jp

Programmed by Yoshiaki WATANABE

**************************************************/
#include "opengatesrv.h"
#include "../ezxml/ezxml.h"

#define CONFIG_VERSION "1.4.11"
#define SEPARATOR "/"

int debug=0;
static ezxml_t xmlRoot=NULL;
static ezxml_t xmlExtraSet=NULL;

char *getConfValueExtra(char *name);
char *getConfValue(char *name);
char *convertToFacilityRaw(char *pValue);

/**************************************************/
/* Prepare Conf file to use                       */
/* this is called before syslog setup             */
/**************************************************/
int openConfFile(void)
{
  char buff[BUFFMAXLN];
  char *s;
  char *errMsg;

  /* parse file and make tree */
  if((xmlRoot = ezxml_parse_file(CONFIGFILE))==NULL){

    /* as the syslog is not prepared, error is send to web*/
    strncpy(buff, "<H3>Error: Opengate configuration file ",BUFFMAXLN);
    strncat(buff, CONFIGFILE,BUFFMAXLN);
    strncat(buff, " is not found. Call the administrator.</H3><BR>",BUFFMAXLN);
     PutClientMsg(buff);

    return -1;
  }

  /* to check error, convert to xml */
  s=ezxml_toxml(xmlRoot);  free(s);
  
  /* if failed, show error message */
  errMsg=(char *)ezxml_error(xmlRoot);

  if(*errMsg!='\0'){
    /* as the syslog is not prepared, error is send to web*/
    strncpy(buff, "<H3>Error: Opengate configuration file ",BUFFMAXLN);
    strncat(buff, CONFIGFILE,BUFFMAXLN);
    strncat(buff, " is illegal. Call the administrator.</H3><HR>",BUFFMAXLN);
    strncat(buff, "XML parser message: ", BUFFMAXLN);
    strncat(buff, errMsg, BUFFMAXLN);
    strncat(buff, "<HR>", BUFFMAXLN);
     PutClientMsg(buff);

    return -1;
  }

  /* check the config file version */ 
  if(isNull(ezxml_attr(xmlRoot, "ConfigVersion"))||
     (strcmp(CONFIG_VERSION, ezxml_attr(xmlRoot, "ConfigVersion"))!=0)){
    strncpy(buff, "<H3>Error: Opengate configuration file ",BUFFMAXLN);
    strncat(buff, CONFIGFILE, BUFFMAXLN);
    strncat(buff, " has mismatch version.<br> Please update it with ",BUFFMAXLN);
    strncat(buff, CONFIGFILE, BUFFMAXLN);
    strncat(buff, ".sample.",BUFFMAXLN);
     PutClientMsg(buff);

    return -1;
  }

  /* check the syslog */
  if(atoi(GetConfValue("Syslog/Enable")) &&
     atoi(GetConfValue("Syslog/Facility"))==0){

    /* as the syslog is not prepared, error is send to web*/
    strncpy(buff, "<H3>Error: correct SYSLOG setting(local0-local7) is not found in Opengate configuration file ",BUFFMAXLN);
    strncat(buff, CONFIGFILE,BUFFMAXLN);
    strncat(buff, ". Call the administrator.</H3><BR>",BUFFMAXLN);
     PutClientMsg(buff);

    return -1;
  }

  return 0;
}

/**************************************************/
/*  initialize the Config                         */
/**************************************************/
void initConf(void)
{
  /* as debug flag is used many times, put it in gloval variable */
  debug=atoi(getConfValue("Debug"));
}

/**************************************************/
/* Finish Conf file usage                         */
/**************************************************/
void closeConfFile(void)
{
  if(xmlRoot!=NULL)ezxml_free(xmlRoot);
}

/**************************************************/
/* Setup pointer to the matched ExtraSet          */ 
/**************************************************/
void setupConfExtra(char * userId,char *extraId)
{
  ezxml_t xml;

  /* init as no ExtraSet */
  xmlExtraSet=NULL;

  if(isNull(extraId)) return;

  /* search the matching extra set (first match is employed) */
  for(xml=ezxml_child(xmlRoot, "ExtraSet"); xml; xml=xml->next){
    
    /* if extraID is not match, go to next ExtraSet */
    if(strcmp(extraId, ezxml_attr(xml, "ExtraId"))!=0) continue;

    /* if userID pattern is exist, check it */
    if(!isNull(ezxml_attr(xml, "UserIdPattern"))){

      /* if not matched, go to next ExtraSet */
      if(RegExMatch(userId, ezxml_attr(xml, "UserIdPattern"))==FALSE)continue;
    }

    /* found matched ExtraSet */
    break;
  }

  /* if found a matched ExtraSet, save the pointer */
  if(xml!=NULL) xmlExtraSet=xml;

  /* change syslog setting */
  errToSyslog(atoi(GetConfValue("Syslog/Enable")));
  openlog(GetConfValue("MainCgi"), LOG_PID, atoi(GetConfValue("Syslog/Facility")));

  /* reset config setting */
  InitConf();
}

/***********************************************/
/* regular expression matching                 */
/*  inStr : string to match                    */
/*  regEx : regular expression                 */
/***********************************************/
int regExMatch(const char *inStr, const char *regEx)
{
  regex_t reg;
  int errcode;
  int match;
  char errbuff[WORDMAXLN];

  /* compile regex */
  if((errcode=regcomp(&reg, regEx, REG_NOSUB|REG_EXTENDED|REG_ICASE))!=0){
    regerror(errcode, &reg, errbuff, WORDMAXLN);
    err_msg("ERR at %s#%d: regex message=%s",__FILE__,__LINE__,errbuff);
    match=FALSE;
  }
  
  /* if compile is success, check the input string */
  else{
    if(regexec(&reg, inStr, (size_t)0, NULL, 0)==0) match=TRUE;
    else match=FALSE;
  }

  regfree(&reg);

  return match;
}

/**************************************************/
/*  get a value for name from Conf file           */
/*  the name[aa/bb/cc] means the path             */
/*  if ID is set, extraSet value is overlayed */
/**************************************************/
char *getConfValue(char *name)
{
  char *pValue;
  char *pValueExtra;
  char *pStr;
  char buff[BUFFMAXLN];
  ezxml_t xml;

  /* copy name to work area */
  strncpy(buff,name,BUFFMAXLN);

  /* get first token */
  pStr=strtok(buff, SEPARATOR);

  /* set search start to root of tree */
  xml=xmlRoot;

  /* search the tree node for the name */
  while(pStr!=NULL){
    xml=ezxml_child(xml, pStr);
    pStr=strtok(NULL, SEPARATOR);
  }

  /* get the node value */
  pValue= ezxml_txt(xml);

  /* if not get, write error message */
  if(isNull(pValue)){
    /* AuthServer parameters (except Protocol) might be omitted */
    if((strstr(name,"AuthServer")==name)&&
       strstr(name,"AuthServer/Protocol")==NULL){
    }else{
      /* other parameters must be set */
      err_msg("ERR at %s#%d: cannot get %s from conf file",__FILE__,__LINE__,name);
    }
  }

  /* get value in extra set matched to ID */
  /* if name is matched in first level, reset all child setting */
  /* in this section, many parameters are not set */
  if(!isNull(pValueExtra=getConfValueExtra(name))){
    pValue=pValueExtra;
  }

  /* if syslog facility, the id is converted to raw value */
  if(strcmp(name,"Syslog/Facility")==0){
    pValue=convertToFacilityRaw(pValue);
  }

  /* return found value */
  return pValue;
}
/************************************************/
/* get the value in extra set matched to ID     */
/************************************************/
char *getConfValueExtra(char *name)
{
  char *pStr;
  char buff[BUFFMAXLN];
  ezxml_t xml;

  if(xmlExtraSet==NULL) return "";

  /* extract first token in name */
  strncpy(buff,name,BUFFMAXLN);
  pStr=strtok(buff, SEPARATOR);  

  /* get a first level matched node in extra set */
  /* the first level is not included in the following loop */
  /* as to prevent partial overlay of sub level value */
  xml=ezxml_child(xmlExtraSet, pStr);
  if(xml==NULL) return "";

  /* search the node matched to name */
  pStr=strtok(NULL, SEPARATOR);
  while(pStr!=NULL){
    xml=ezxml_child(xml, pStr);
    pStr=strtok(NULL, SEPARATOR);
  }

  /* return the found value */
  return ezxml_txt(xml);
}

/***********************************************/
/* Convart the syslog facility id to raw value */
/***********************************************/
char *convertToFacilityRaw(char *pValue)
{
  static char facility[WORDMAXLN];
  int rawValue;

  if     (strcmp(pValue, "local0")==0) rawValue=LOG_LOCAL0;
  else if(strcmp(pValue, "local1")==0) rawValue=LOG_LOCAL1;
  else if(strcmp(pValue, "local2")==0) rawValue=LOG_LOCAL2;
  else if(strcmp(pValue, "local3")==0) rawValue=LOG_LOCAL3;
  else if(strcmp(pValue, "local4")==0) rawValue=LOG_LOCAL4;
  else if(strcmp(pValue, "local5")==0) rawValue=LOG_LOCAL5;
  else if(strcmp(pValue, "local6")==0) rawValue=LOG_LOCAL6;
  else if(strcmp(pValue, "local7")==0) rawValue=LOG_LOCAL7;
  else rawValue=0;

  snprintf(facility, WORDMAXLN, "%d", rawValue);

  return facility;
}

/***********************************************/
/***********************************************/
int OpenConfFile(void){
  int ret;
  if(debug>1) err_msg("DEBUG:=>openConfFile( )");
  ret = openConfFile();
  if(debug>1) err_msg("DEBUG:(%d)<=openConfFile( )",ret);
  return ret;
}

void CloseConfFile(void){
  if(debug>1) err_msg("DEBUG:=>closeConfFile( )");
  closeConfFile();
  if(debug>1) err_msg("DEBUG:<=closeConfFile( )");
}

void SetupConfExtra(char *userId, char *extraId){
  if(debug>1) err_msg("DEBUG:=>setupConfExtra(%s,%s)",userId, extraId);
  setupConfExtra(userId, extraId);
  if(debug>1) err_msg("DEBUG:<=setupConfExtra( )");
}

char *GetConfValue(char *name){
  char *ret;
  if(debug>1) err_msg("DEBUG:=>getConfValue(%s)",name);
  ret=getConfValue(name);
  if(debug>1) err_msg("DEBUG:(%s)<=getConfValue( )",ret);
  return ret;
}

char *GetConfValueExtra(char *name){
  char *ret;
  if(debug>1) err_msg("DEBUG:=>getConfValueExtra(%s)",name);
  ret=getConfValueExtra(name);
  if(debug>1) err_msg("DEBUG:(%s)<=getConfValueExtra( )",ret);
  return ret;
}

void InitConf(void){
  if(debug>1) err_msg("DEBUG:=>initConf( )");
  initConf();
  if(debug>1) err_msg("DEBUG:<=initConf( )");
}

int RegExMatch(const char *inStr, const char *regEx){
  int ret;
  if(debug>1) err_msg("DEBUG:=>regExMatch(%s,%s)", inStr, regEx);
  ret=regExMatch(inStr, regEx);
  if(debug>1) err_msg("DEBUG:(%d)<=regExMatch( )",ret);
  return ret;
}
  
