/**************************************************
Opengate server
 module for Communication through CGI 

Copyright (C) 1999 Opengate Project Team
Written by Yoshiaki Watanabe
Modified Katsuhiko Eguchi, 2005 

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Email: watanaby@is.saga-u.ac.jp

Programmed by Yoshiaki WATANABE
Modified by Shin-ichi TADAKI
Modified by Katsuhiko Eguchi
**************************************************/

#include	"opengatesrv.h"

/* convert two-char-hex "aa" to one-number 0Xaa */ 
#define hex2num(x)  ((x)>='A' ? ((x) & 0XDF) - 'A' +10 : ((x) - '0'))

int isHttpWatchEnableClient(void);
int isJavaWatchEnableClient(void);
void split(char content[], char *name[], char *value[], char *next[]);
void decode(char *string);

char language[WORDMAXLN]; /* message language in java applet */

/*******************************/
/* get the client addr         */
/*******************************/
void getClientAddr(char *clientAddr)
{
  strncpy(clientAddr, getenv("REMOTE_ADDR"), ADDRMAXLN);
}

/********************************************/
/* get Post data from the client  */
/********************************************/
int getPostData(char *userid, char *password, char *clientAddr4, int *durationPtr, int *durationEntered)
{
  int contentLen;
  char content[BUFFMAXLN];
  char *name[1];
  char *value[1];
  char *next[1];
  char *ptr;
  char durationStr[WORDMAXLN];
  char langList[BUFFMAXLN];
  char encodeAddr4[ADDRMAXLN];
  char accessAddr[ADDRMAXLN];

  /* get content sent from web input */
  if(getenv("CONTENT_LENGTH")==NULL){
    err_msg("ERR at %s#%d: CONTENT_LENGTH is not defined",__FILE__,__LINE__);
    return FALSE;
  }

  contentLen=atoi(getenv("CONTENT_LENGTH"));
  if(contentLen==0){
    err_msg("ERR at %s#%d: CONTENT_LENGTH is zero",__FILE__,__LINE__);
    return FALSE;
  }

  contentLen++; /* for terminate ch */
  if(contentLen > BUFFMAXLN) contentLen=BUFFMAXLN;
  if(fgets(content, contentLen, stdin) == NULL){
    content[0]='\0';
  }

  /* get items from string */
  userid[0]='\0';
  password[0]='\0';
  encodeAddr4[0]='\0';
  language[0]='\0';
  durationStr[0]='\0';

  ptr=content;

  while(ptr!=NULL){
    split(ptr, name, value, next);

    if(strstr(name[0], "userid")!=NULL){
      strncpy(userid, value[0], USERMAXLN);
    }else if(strstr(name[0], "password")!=NULL){
      strncpy(password, value[0], PASSMAXLN);
    }else if(strstr(name[0],"remote_addr")!=NULL){
      strncpy(encodeAddr4,value[0],ADDRMAXLN);
    }else if(strstr(name[0], "language")!=NULL){
      strncpy(language, value[0], WORDMAXLN);
    }else if(strstr(name[0], "duration")!=NULL){
      strncpy(durationStr, value[0], WORDMAXLN);
    }
    ptr=next[0];
  }

  /* decode the HTTP encoding */
  decode(userid);
  decode(password);
  decode(encodeAddr4);
  decode(language);
  decode(durationStr);

 /* if not available language, use first lang */
  strncpy(langList, GetConfValue("HtmlLangs"), BUFFMAXLN);  /* list of available languages */
  if(strstr(langList,language)==NULL){
    sscanf(langList,"%s",language);
  }
  
  /* convert duration string to interger and minutes to seconds */
  *durationPtr = atoi(durationStr)*60;
  *durationEntered = TRUE;

  /* usage duration is restricted to permitted range */
  if(*durationPtr <= 0){
    *durationEntered = FALSE;
    *durationPtr= atoi(GetConfValue("Duration/Default"));
  }else{
    int durmax=atoi(GetConfValue("Duration/Max"));
    if(*durationPtr > durmax) *durationPtr=durmax;
  }

  /* encoded address starting as "0-0-0" means no addr info */
  /* it indicates needless to get dual stack addresses */
  /* and only use getenv("REMOTE_ADDR") address */
  if(strnstr(encodeAddr4, "0-0-0", ADDRMAXLN)==encodeAddr4){
    clientAddr4[0]='\0';
  }
      
  /* decode client address to dot separated form */
  else if(AddrDecode(clientAddr4, encodeAddr4)==1){
    /* if can't decode, retry */
    err_msg("ERR at %s#%d: Cannot decode client address",__FILE__,__LINE__);
    return FALSE;
  }

  /* if the decoded IPv4 addr is not same as access IPv4 addr, use later */
  strncpy(accessAddr, getenv("REMOTE_ADDR"), ADDRMAXLN);
  if((strnstr(accessAddr, ".", ADDRMAXLN)!=NULL)   /* access is IPv4 */
     && strncmp(accessAddr, clientAddr4, ADDRMAXLN)!=0){  /* and not same */
    strncpy(clientAddr4, accessAddr, ADDRMAXLN);
  }
  return TRUE;
}


/*********************************************/
/* deny message to the client            */
/*********************************************/
void putClientDeny(char *clientAddr4)
{
  char denydoc[BUFFMAXLN];
  char authCgiUrl[BUFFMAXLN];
  char encodeAddr[ADDRMAXLN];
  char opengateDir[BUFFMAXLN];

  /* keyword pairs */
  /*  the left key is replaced by the right value */
  struct html_key keys[]=
    {
      {"%%OPENGATEDIR%%", opengateDir},
      {"%%AUTHCGIURL%%", authCgiUrl},
      {"%%ADDR4%%", encodeAddr},
      {"",""}  /* DON'T REMOVE THIS LINE */
    };
 
  /* create authcgi URL string */
  snprintf(authCgiUrl, BUFFMAXLN, "%s%s%s/%s",
	   GetConfValue("OpengateServerName"),
	   GetConfValue("CgiDir"),
	   GetConfValue("OpengateDir"),
	   GetConfValue("AuthCgi"));

  /* create opengate dir */
  snprintf(opengateDir, BUFFMAXLN, GetConfValue("OpengateDir"));

  /* create encoded addr4 */
  if(AddrEncode(encodeAddr, clientAddr4)==1){
    encodeAddr[0]='\0';
  }

  /* make path to the denydoc for ssl or non-ssl */
  if(strcmp(getenv("SERVER_PORT"),GetServicePortStr("https"))==0){
    snprintf(denydoc, BUFFMAXLN, "%s%s/%s/%s",GetConfValue("DocumentRoot"),
	    GetConfValue("OpengateDir"),language,GetConfValue("DenyDocSsl"));
  }else{
    snprintf(denydoc, BUFFMAXLN, "%s%s/%s/%s",GetConfValue("DocumentRoot"),
	    GetConfValue("OpengateDir"),language,GetConfValue("DenyDoc"));
  }

  /* replace keyword and send out the file */
  printf("Content-type: text/html\r\n\r\n\r\n");
  HtmlTemplate(denydoc, keys);

  return;
}
/*********************************************/
/* deny message to the client            */
/*********************************************/
void putClientRetry(char *lang)
{
  char retrydoc[BUFFMAXLN];
  char externalUrl[BUFFMAXLN];
  char authCgiUrl[BUFFMAXLN];
  char opengateDir[BUFFMAXLN];

  /* keyword pairs */
  /*  the left key is replaced by the right value */
  struct html_key keys[]=
    {
      {"%%OPENGATEDIR%%", opengateDir},
      {"%%EXTERNALURL%%", externalUrl},
      {"%%AUTHCGIURL%%", authCgiUrl},
      {"",""}  /* DON'T REMOVE THIS LINE */
    };

  /* create opengate Dir */
  snprintf(opengateDir, BUFFMAXLN, GetConfValue("OpengateDir"));
 
  /* create external URL string */
  strncpy(externalUrl, GetConfValue("ExternalUrl"), BUFFMAXLN);

  /* create authcgi URL string */
  snprintf(authCgiUrl, BUFFMAXLN, "%s%s%s/%s",
	   GetConfValue("OpengateServerName"),
	   GetConfValue("CgiDir"),
	   opengateDir,
	   GetConfValue("AuthCgi"));

  /* make read in path to the retry document */
  snprintf(retrydoc, BUFFMAXLN, "%s%s/%s/%s",GetConfValue("DocumentRoot"),
	  opengateDir,lang,GetConfValue("RetryDoc"));

  /* replace keyword and send out the file */
  printf("Content-type: text/html\r\n\r\n\r\n");
  HtmlTemplate(retrydoc, keys);

  return;
}

/*********************************************/
/* put some message to the client            */
/*********************************************/
void putClientMsg(char *message)
{
  printf("Content-type: text/html\r\n\r\n\r\n");
  printf("<HTML><HEAD><TITLE>OpengateMsg</TITLE></HEAD> \r\n");
  printf("<BODY>         \r\n");
  printf("%s\r\n",     message);
  printf("</BODY></HTML> \r\n\r\n");

}

/*********************************************/
/* put accept message and java to the client */
/*********************************************/
void putClientAccept(char *userid, char *sessionId, int port, int pid, char *clientAddr4, char *clientAddr6, int ipStatus, int duration, int durationEntered)
{
  FILE *fp;
  char buff[BUFFMAXLN];
  char acceptdoc[BUFFMAXLN];
  char acceptdoc2url[BUFFMAXLN];
  char terminateurl[BUFFMAXLN];
  char httpkeepUrl[BUFFMAXLN];
  char portStr[WORDMAXLN];
  char durationStr[WORDMAXLN];
  char *acceptDoc;

  char *startPageUrl=GetConfValue("StartPage/Url");
  int startPageType=atoi(GetConfValue("StartPage/Type"));
  char *opengateDir=GetConfValue("OpengateDir");
  char *opengateServerName=GetConfValue("OpengateServerName");

  /* select proper accept doc */
  switch(toupper(*GetConfValue("WatchMode"))){

    /* HTTP watch mode */
  case 'H':
  default:
    if(isHttpWatchEnableClient()){
      acceptDoc=GetConfValue("AcceptDocHttp");
    }else if(isJavaWatchEnableClient()){
      acceptDoc=GetConfValue("AcceptDocJava");
    }else{
      acceptDoc=GetConfValue("AcceptDocTime");
    }
    break;

    /* JAVA watch mode */
  case 'J':
    if(isJavaWatchEnableClient()){
      acceptDoc=GetConfValue("AcceptDocJava");
    }else if(isHttpWatchEnableClient()){
      acceptDoc=GetConfValue("AcceptDocHttp");
    }else{
      acceptDoc=GetConfValue("AcceptDocTime");
    }
    break;

    /* TIMEOUT watch mode */
  case 'T':
    acceptDoc=GetConfValue("AcceptDocTime");
    break;
  }

  /* if positive value is set in duration, TIME watch mode is selected */
  if(durationEntered) acceptDoc=GetConfValue("AcceptDocTime");

  /* create path to acceptdoc */
  snprintf(acceptdoc, BUFFMAXLN, "%s%s/%s/%s",GetConfValue("DocumentRoot"),
	 GetConfValue("OpengateDir"),language,acceptDoc);


  snprintf(acceptdoc2url, BUFFMAXLN, 
	  "http://%s%s/%s/%s",GetConfValue("OpengateServerName"),
	  GetConfValue("OpengateDir"),language,GetConfValue("AcceptDoc2"));

  /* create terminate url [http://<servaddr>:<port>/terminate-<pid>] */
  snprintf(terminateurl, BUFFMAXLN, "http://%s:%d/terminate-%d", 
	  GetConfValue("OpengateServerName"), port, getpid());

  /* create httpkeep page url
     ['http://<servaddr>:<port>/httpkeep-<userid>'] */
  snprintf(httpkeepUrl, BUFFMAXLN,
	   "'http://%s:%d/httpkeep-%s-%s'", 
	  GetConfValue("OpengateServerName"), port, userid,sessionId);

  /* create port string */
  snprintf(portStr, WORDMAXLN, "%d", port);

  /* create duration string (duration=sec, display value=min) */
  snprintf(durationStr, WORDMAXLN, "%d", duration/60);

  /* open acceptdoc */
  if((fp=fopen(acceptdoc, "r"))==NULL){
    err_msg("ERR at %s#%d: cannot open %s",__FILE__,__LINE__,acceptdoc);
    PutClientMsg("Cannot find html document");
    return;
  }

  /* read html document from file and send to web */
  printf("Content-type: text/html\r\n\r\n\r\n");
  while(fgets(buff, BUFFMAXLN, fp)!=NULL){

    /* length check */
    if(strlen(buff)>=BUFFMAXLN-1){
      err_msg("ERR at %s#%d: too long line in %s",__FILE__,__LINE__,acceptdoc);
    }

    /* replace mark */
    htmlReplace(buff, "%%OPENGATESERVERNAME%%", opengateServerName);
    htmlReplace(buff, "%%OPENGATEDIR%%", opengateDir);
    htmlReplace(buff, "%%OPENGATEPORT%%", portStr);
    htmlReplace(buff, "%%DURATION%%", durationStr);
    htmlReplace(buff, "%%USERID%%", userid);
    htmlReplace(buff, "%%SESSIONID%%", sessionId);
    htmlReplace(buff, "%%LANGUAGE%%", language);
    
    htmlReplace(buff, "%%TERMINATEURL%%", terminateurl);
    htmlReplace(buff, "%%HTTPKEEPURL%%", httpkeepUrl);
    
    /* replace information url mark */
    if( startPageType==1 ){
      htmlReplace(buff, "%%STARTURL%%", startPageUrl);
    }else{
      htmlReplace(buff, "%%STARTURL%%", acceptdoc2url);
    }
    
    /* write out */
    printf("%s",buff);
  }
  /* write end */
  fputs("\r\n\r\n",stdout);
  fclose(fp);

  return;
}

/*****************************************************/
/* is the client enable to keep long http connection */
/*****************************************************/
int isHttpWatchEnableClient(void)
{
  /* HTTP Keep-Alive is not standard in http/1.0 */
  if(strcmp(getenv("SERVER_PROTOCOL"),"HTTP/1.0")==0) return FALSE;

  /* some user agent does not support long HTTP Keep-Alive */
  if(RegExMatch(getenv("HTTP_USER_AGENT"),
		 GetConfValue("HttpWatch/SkipAgentPattern"))) return FALSE;

  return TRUE;
}

/********************************************/
/* is the client enable to load Java Applet */
/********************************************/
int isJavaWatchEnableClient(void)
{
  /* some user agent does not support Java Applet */
  if(RegExMatch(getenv("HTTP_USER_AGENT"),
		 GetConfValue("JavaWatch/SkipAgentPattern"))) return FALSE;

  return TRUE;
}

/************************************/
/* split value for indicated name   */
/* in content  "name=value&..."     */
/************************************/
void split(char content[], char *name[], char *value[], char *next[])
{
  char *pstr;
  
  /* set default */
  name[0]=content;
  value[0]=content+strlen(content);
  next[0]=value[0];

  /* set name end */
  if((pstr=strchr(name[0],(int)'='))==NULL){
    next[0]=NULL;
    return;
  }
  *pstr='\0';
  
  /* set value start */
  pstr++;
  value[0]=pstr;
  
  /* set value end */
  if((pstr=strchr(value[0],'&'))==NULL){
    next[0]=NULL;
    return;
  }
  *pstr='\0';

  /* set next start */
  pstr++;
  next[0]=pstr;

  return;
}

/**********************************/
/* decode text coding in web post */
/**********************************/
void decode(char *string)
{
  char *pcheck, *pinsert;

  pcheck=pinsert=string;
  while(*pcheck != '\0'){
    if(*pcheck == '+'){
      *pinsert = ' ';
    }else if(*pcheck == '%'){
      *pinsert=(char)(hex2num(*(pcheck+1))*16 + hex2num(*(pcheck+2)));
      pcheck+=2;
    }else{
      *pinsert=*pcheck;
    }
    pcheck++;
    pinsert++;
  }
  *pinsert=*pcheck;
}

/***************************************/
/* get HTTP_REFERER and check true url */
/***************************************/
int checkReferer(void)
{
  char url[BUFFMAXLN]="";
  if(getenv("HTTP_REFERER")!=NULL){
    strncpy(url,getenv("HTTP_REFERER"),BUFFMAXLN);
    if(strstr(url,GetConfValue("OpengateServerName"))==NULL){
      return FALSE;
    }
  }
  return TRUE;
}

/*******************************/
/*******************************/
void GetClientAddr(char *clientAddr)
{
  if(debug>1) err_msg("DEBUG:=>getClientAddr( )");
  getClientAddr(clientAddr);
  if(debug>1) err_msg("DEBUG:<=getClientAddr(%s)",clientAddr);
}


int GetPostData(char *userid, char *password, char *clientAddr4, int *durationPtr, int *durationEntered)
{
  int ret;

  if(debug>1) err_msg("DEBUG:=>getPostData( )");
  ret=getPostData(userid,password,clientAddr4,durationPtr,durationEntered);
  if(debug>1) err_msg("DEBUG:%d<=getPostData(%s,passwd,%s,%d,%d)",ret,userid,clientAddr4,*durationPtr,durationEntered);
  return ret;
}

void PutClientAccept(char *userid, char *sessionId, int port, int pid, char *clientAddr4, char *clientAddr6, int ipStatus, int duration, int durationEntered)
{
  if(debug>1) err_msg("DEBUG:=>putClientAccept(%s,%s,%d,%d,%s,%s,%d,%d,%d)",userid,sessionId,port,pid,clientAddr4,clientAddr6,ipStatus, duration, durationEntered);
  putClientAccept(userid,sessionId,port,pid,clientAddr4,clientAddr6,ipStatus,duration, durationEntered);
  if(debug>1) err_msg("DEBUG:<=putClientAccept( )");
}

void PutClientDeny(char *clientAddr4)
{
  if(debug>1) err_msg("DEBUG:=>putClientDeny(&s)",clientAddr4);
  putClientDeny(clientAddr4);
  if(debug>1) err_msg("DEBUG:<=putClientDeny( )");
}

void PutClientRetry(char *lang)
{
  if(debug>1) err_msg("DEBUG:=>putClientRetry(%s)",lang);
  putClientRetry(lang);
  if(debug>1) err_msg("DEBUG:<=putClientRetry( )");
}

void PutClientMsg(char *message)
{
  if(debug>1) err_msg("DEBUG:=>putClientMsg( %s )",message);
  putClientMsg(message);
  if(debug>1) err_msg("DEBUG:<=putClientMsg( )");
}

int CheckReferer(void)
{
  int ret;
  if(debug>1) err_msg("DEBUG:=>checkReferer( )");
  ret = checkReferer();
  if(debug>1) err_msg("DEBUG:(%d)<=checkReferer( )",ret);
  return ret;
}
